package ru.yandex.direct.internaltools.tools.idm.tool;

import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.common.spring.TestingComponent;
import ru.yandex.direct.core.entity.idm.model.IdmGroup;
import ru.yandex.direct.core.entity.idm.model.IdmRequiredRole;
import ru.yandex.direct.core.entity.idm.repository.IdmGroupsRepository;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.idm.model.IdmGroupAddParameters;
import ru.yandex.direct.internaltools.tools.idm.model.IntToolIdmGroup;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.defect.CommonDefects.unknownParameter;

@Tool(
        name = "1 Добавить IDM-группу",
        label = "add_idm_group",
        description = "Добавление IDM-группы в ppcdict.idm_groups. \n Этот отчёт не предполагается " +
                "использовать в продакшене, только для тестов.",
        consumes = IdmGroupAddParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.ADD)
@Category(InternalToolCategory.IDM)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
@TestingComponent
public class AddIdmGroupTool extends MassInternalTool<IdmGroupAddParameters, IntToolIdmGroup> {

    private static final Set<String> ROLE_NAMES = StreamEx.of(IdmRequiredRole.values())
            .map(IdmRequiredRole::getTypedValue)
            .toSet();

    private final IdmGroupsRepository idmGroupsRepository;

    public AddIdmGroupTool(IdmGroupsRepository idmGroupsRepository) {
        this.idmGroupsRepository = idmGroupsRepository;
    }

    @Override
    public ValidationResult<IdmGroupAddParameters, Defect> validate(IdmGroupAddParameters params) {
        ItemValidationBuilder<IdmGroupAddParameters, Defect> vb =
                ItemValidationBuilder.of(params, Defect.class);
        vb.item(params.getId(), IdmGroupAddParameters.ID_PROP_NAME)
                .check(notNull())
                .check(validId(), When.isValid())
                .checkBy(this::validateGroupNotExist, When.isValid());
        vb.item(params.getRoleName(), IdmGroupAddParameters.ROLE_NAME_PROP_NAME)
                .check(notNull())
                .check(notBlank(), When.isValid())
                .check(inSet(ROLE_NAMES), unknownParameter(), When.isValid());
        return vb.getResult();
    }

    private ValidationResult<Long, Defect> validateGroupNotExist(Long idxGroupId) {
        List<IdmGroup> groupsRoles = idmGroupsRepository.getGroups(Collections.singleton(idxGroupId));
        if (groupsRoles.isEmpty()) {
            return ValidationResult.success(idxGroupId);
        }
        return ValidationResult.failed(idxGroupId, CommonDefects.inconsistentStateAlreadyExists());
    }

    @Nullable
    @Override
    protected List<IntToolIdmGroup> getMassData() {
        List<IdmGroup> groupsRoles = idmGroupsRepository.getAllGroupsRoles();
        return mapList(groupsRoles, this::toIntToolIdmGroup);
    }

    private IntToolIdmGroup toIntToolIdmGroup(IdmGroup idmGroup) {
        return new IntToolIdmGroup()
                .withId(idmGroup.getIdmGroupId())
                .withRoleName(idmGroup.getRequiredRole().getTypedValue());
    }

    @Override
    protected List<IntToolIdmGroup> getMassData(IdmGroupAddParameters parameters) {
        IdmGroup idmGroup = toIdmGroup(parameters);
        idmGroupsRepository.add(Collections.singleton(idmGroup));
        return getMassData();
    }

    private IdmGroup toIdmGroup(IdmGroupAddParameters parameters) {
        return new IdmGroup()
                .withIdmGroupId(parameters.getId())
                .withRequiredRole(IdmRequiredRole.fromTypedValue(parameters.getRoleName()));
    }

}
