package ru.yandex.direct.internaltools.tools.idm.tool;

import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.common.spring.TestingComponent;
import ru.yandex.direct.core.entity.idm.model.IdmGroupMember;
import ru.yandex.direct.core.entity.idm.service.IdmGroupsMembersService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.idm.model.IdmGroupMemberParameters;
import ru.yandex.direct.internaltools.tools.idm.model.IntToolIdmGroupMember;
import ru.yandex.direct.internaltools.tools.idm.validation.IdmGroupMemberParametersValidationService;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.internaltools.tools.idm.validation.IdmGroupMemberParametersValidationService.isAddOperation;
import static ru.yandex.direct.internaltools.tools.idm.validation.IdmGroupMemberParametersValidationService.isDeleteOperation;
import static ru.yandex.direct.internaltools.tools.idm.validation.IdmGroupMemberParametersValidationService.isShowOperation;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Tool(
        name = "2 Управление членством в IDM-группах",
        label = "manage_idm_group_members",
        description = "Для добавления внутреннего пользователя в IDM-группу пользователь должен иметь роль " +
                "соответствующую роли группы. \n Этот отчёт не предполагается использовать в продакшене, только для " +
                "тестов.",
        consumes = IdmGroupMemberParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.IDM)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
@TestingComponent
public class ManageIdmGroupMembersTool extends MassInternalTool<IdmGroupMemberParameters, IntToolIdmGroupMember> {

    private final UserService userService;
    private final IdmGroupsMembersService idmGroupsMembersService;
    private final IdmGroupMemberParametersValidationService validationService;

    public ManageIdmGroupMembersTool(UserService userService,
                                     IdmGroupsMembersService idmGroupsMembersService,
                                     IdmGroupMemberParametersValidationService validationService) {
        this.userService = userService;
        this.idmGroupsMembersService = idmGroupsMembersService;
        this.validationService = validationService;
    }

    @Override
    public ValidationResult<IdmGroupMemberParameters, Defect> validate(IdmGroupMemberParameters params) {
        return validationService.validate(params);
    }

    @Nullable
    @Override
    protected List<IntToolIdmGroupMember> getMassData() {
        return mapList(idmGroupsMembersService.getAllMembers(), this::toIntToolIdmGroupMember);
    }

    private IntToolIdmGroupMember toIntToolIdmGroupMember(IdmGroupMember member) {
        return new IntToolIdmGroupMember()
                .withClientId(member.getClientId())
                .withDomainLogin(member.getDomainLogin())
                .withIdmGroupId(member.getIdmGroupId())
                .withLogin(member.getLogin())
                .withUid(member.getUid());
    }

    @Override
    protected List<IntToolIdmGroupMember> getMassData(IdmGroupMemberParameters params) {
        if (isAddOperation(params)) {
            IdmGroupMember member = toIdmGroupMember(params);
            idmGroupsMembersService.addMembersWhichNotExist(singletonList(member));
        } else if (isDeleteOperation(params)) {
            User user = userService.getUserByLogin(params.getLogin());
            idmGroupsMembersService.removeMember(user.getClientId(), params.getIdmGroupId());
        }
        List<IntToolIdmGroupMember> massData = getMassData();
        if (isShowOperation(params)) {
            if (params.getIdmGroupId() != null) {
                massData = filterList(massData, m -> Objects.equals(m.getIdmGroupId(), params.getIdmGroupId()));
            }
            if (params.getLogin() != null) {
                massData = filterList(massData, m -> Objects.equals(m.getLogin(), params.getLogin()));
            }
        }
        return massData;
    }

    private IdmGroupMember toIdmGroupMember(IdmGroupMemberParameters parameters) {
        User user = userService.getUserByLogin(parameters.getLogin());
        //noinspection ConstantConditions
        return new IdmGroupMember()
                .withIdmGroupId(parameters.getIdmGroupId())
                .withClientId(user.getClientId())
                .withLogin(user.getLogin())
                .withDomainLogin(user.getDomainLogin())
                .withUid(user.getUid());
    }

}
