package ru.yandex.direct.internaltools.tools.idm.tool;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.common.spring.TestingComponent;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.idm.model.IdmGroupRole;
import ru.yandex.direct.core.entity.idm.service.IdmGroupsRolesService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.idm.model.IdmGroupRoleParameters;
import ru.yandex.direct.internaltools.tools.idm.model.IntToolIdmGroupRole;
import ru.yandex.direct.internaltools.tools.idm.validation.IdmGroupRoleParametersValidationService;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.internaltools.tools.idm.validation.IdmGroupRoleParametersValidationService.isAddOperation;
import static ru.yandex.direct.internaltools.tools.idm.validation.IdmGroupRoleParametersValidationService.isDeleteOperation;
import static ru.yandex.direct.internaltools.tools.idm.validation.IdmGroupRoleParametersValidationService.isShowOperation;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Tool(
        name = "3 Управление привязкой клиентов к IDM-группам",
        label = "manage_idm_group_roles",
        description = "Один клиент может быть привязан к нескольким группам. \n Этот отчёт не предполагается " +
                "использовать в продакшене, только для тестов.",
        consumes = IdmGroupRoleParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.IDM)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
@TestingComponent
public class ManageIdmGroupRolesTool extends MassInternalTool<IdmGroupRoleParameters, IntToolIdmGroupRole> {

    private final ClientService clientService;
    private final IdmGroupsRolesService idmGroupsRolesService;
    private final IdmGroupRoleParametersValidationService validationService;

    public ManageIdmGroupRolesTool(ClientService clientService,
                                   IdmGroupsRolesService idmGroupsRolesService,
                                   IdmGroupRoleParametersValidationService validationService) {
        this.clientService = clientService;
        this.idmGroupsRolesService = idmGroupsRolesService;
        this.validationService = validationService;
    }

    @Override
    public ValidationResult<IdmGroupRoleParameters, Defect> validate(IdmGroupRoleParameters params) {
        return validationService.validate(params);
    }

    @Nullable
    @Override
    protected List<IntToolIdmGroupRole> getMassData() {
        return emptyList();
    }

    private List<IntToolIdmGroupRole> toExternalRoles(List<IdmGroupRole> allRoles) {
        List<ClientId> clientIds = mapList(allRoles, IdmGroupRole::getClientId);
        Collection<Client> clients = clientService.massGetClient(clientIds);
        Map<Long, Client> clientsByIds = listToMap(clients, Client::getClientId);
        return mapList(allRoles, subj -> toIntToolIdmGroupRole(subj, clientsByIds));
    }

    private IntToolIdmGroupRole toIntToolIdmGroupRole(IdmGroupRole role, Map<Long, Client> clientsByIds) {
        ClientId clientId = role.getClientId();
        String name = Optional.ofNullable(clientsByIds.get(clientId.asLong()))
                .map(Client::getName)
                .orElse("");
        return new IntToolIdmGroupRole()
                .withClientId(clientId)
                .withIdmGroupId(role.getIdmGroupId())
                .withName(name);
    }

    @Override
    protected List<IntToolIdmGroupRole> getMassData(IdmGroupRoleParameters params) {
        if (isAddOperation(params)) {
            IdmGroupRole role = toIdmGroupRole(params);
            idmGroupsRolesService.addRolesWhichNotExist(singletonList(role));
            // роль успешно выдали, покажем её в ответе
            Optional<IdmGroupRole> addedRole =
                    idmGroupsRolesService.getRole(ClientId.fromLong(params.getClientId()), params.getIdmGroupId());
            List<IdmGroupRole> roles = addedRole.map(Collections::singletonList).orElse(emptyList());
            return toExternalRoles(roles);
        } else if (isDeleteOperation(params)) {
            idmGroupsRolesService.removeRole(ClientId.fromLong(params.getClientId()), params.getIdmGroupId());
        }
        if (isShowOperation(params)) {
            List<IdmGroupRole> roles;
            if (params.getClientId() != null) {
                roles = idmGroupsRolesService.getRolesByClientId(ClientId.fromLong(params.getClientId()));
                if (params.getIdmGroupId() != null) {
                    roles = filterList(roles, m -> params.getIdmGroupId().equals(m.getIdmGroupId()));
                }
            } else if (params.getIdmGroupId() != null) {
                roles = idmGroupsRolesService.getRolesByGroupId(params.getIdmGroupId());
            } else {
                roles = emptyList();
            }
            return toExternalRoles(roles);
        }
        return emptyList();
    }

    private IdmGroupRole toIdmGroupRole(IdmGroupRoleParameters parameters) {
        return new IdmGroupRole()
                .withIdmGroupId(parameters.getIdmGroupId())
                .withClientId(ClientId.fromLong(parameters.getClientId()));
    }

}
