package ru.yandex.direct.internaltools.tools.idm.validation;

import java.util.Collections;
import java.util.Objects;

import javax.annotation.Nullable;

import com.google.common.collect.Iterables;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.idm.model.IdmGroup;
import ru.yandex.direct.core.entity.idm.model.IdmGroupMember;
import ru.yandex.direct.core.entity.idm.repository.IdmGroupsRepository;
import ru.yandex.direct.core.entity.idm.service.IdmGroupsMembersService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.internaltools.tools.idm.model.IdmGroupMemberParameters;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.StringConstraints;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Service
public class IdmGroupMemberParametersValidationService {

    private final UserService userService;
    private final IdmGroupsRepository idmGroupsRepository;
    private final IdmGroupsMembersService idmGroupsMembersService;

    public IdmGroupMemberParametersValidationService(UserService userService, IdmGroupsRepository idmGroupsRepository,
                                                     IdmGroupsMembersService idmGroupsMembersService) {
        this.userService = userService;
        this.idmGroupsRepository = idmGroupsRepository;
        this.idmGroupsMembersService = idmGroupsMembersService;
    }

    public static boolean isAddOperation(IdmGroupMemberParameters params) {
        return Objects.equals(IdmGroupMemberParameters.ADD_OPERATION, params.getOperation());
    }

    public static boolean isDeleteOperation(IdmGroupMemberParameters params) {
        return Objects.equals(IdmGroupMemberParameters.DELETE_OPERATION, params.getOperation());
    }

    public static boolean isShowOperation(IdmGroupMemberParameters params) {
        return !isAddOperation(params) && !isDeleteOperation(params);
    }

    public ValidationResult<IdmGroupMemberParameters, Defect> validate(IdmGroupMemberParameters params) {
        ItemValidationBuilder<IdmGroupMemberParameters, Defect> vb =
                ItemValidationBuilder.of(params, Defect.class);
        String login = params.getLogin();
        User user = (isNotBlank(login)) ? userService.getUserByLogin(login) : null;
        if (isAddOperation(params)) {
            addOperationValidation(params, vb, user);
        } else if (isDeleteOperation(params)) {
            deleteOperationValidation(params, vb, user);
        }
        return vb.getResult();
    }

    private void deleteOperationValidation(IdmGroupMemberParameters params,
                                           ItemValidationBuilder<IdmGroupMemberParameters,
                                                   Defect> vb, User user) {
        Long idmGroupId = params.getIdmGroupId();
        IdmGroupMember member = (idmGroupId != null && user != null) ?
                idmGroupsMembersService.getMember(user.getClientId(), idmGroupId).orElse(null) : null;
        vb.item(idmGroupId, IdmGroupMemberParameters.ID_PROP_NAME)
                .check(notNull())
                .check(validId(), When.isValid());
        vb.item(params.getLogin(), IdmGroupMemberParameters.LOGIN_PROP_NAME)
                .check(StringConstraints.notBlank());
        if (member == null) {
            vb.getResult().addError(CommonDefects.objectNotFound());
        }
    }

    private void addOperationValidation(IdmGroupMemberParameters params,
                                        ItemValidationBuilder<IdmGroupMemberParameters, Defect> vb, User user) {
        Long idmGroupId = params.getIdmGroupId();
        IdmGroup group = (idmGroupId != null)
                ? Iterables.getFirst(idmGroupsRepository.getGroups(Collections.singleton(idmGroupId)), null)
                : null;
        IdmGroupMember member = (idmGroupId != null && user != null) ?
                idmGroupsMembersService.getMember(user.getClientId(), idmGroupId).orElse(null) : null;
        ItemValidationBuilder<Long, Defect> groupCheck = vb.item(idmGroupId,
                IdmGroupMemberParameters.ID_PROP_NAME)
                .check(notNull())
                .check(validId(), When.isValid())
                .checkBy(getGroupValidator(group), When.isValid());
        vb.item(params.getLogin(), IdmGroupMemberParameters.LOGIN_PROP_NAME)
                .check(StringConstraints.notBlank())
                .checkBy(getUserValidator(user), When.isValid())
                .checkBy(getRightsValidator(group, user), When.isValidBoth(groupCheck));
        if (member != null) {
            vb.getResult().addError(CommonDefects.inconsistentStateAlreadyExists());
        }
    }

    private Validator<? extends String, Defect> getRightsValidator(@Nullable IdmGroup group, @Nullable User user) {
        return login ->
        {
            if (group == null || user == null) {
                return ValidationResult.success(login);
            }
            String userRole = user.getRole().name();
            String requiredRole = group.getRequiredRole().name();
            if (!Objects.equals(userRole, requiredRole)) {
                return ValidationResult.failed(login, CommonDefects.inconsistentState());
            }
            return ValidationResult.success(login);
        };
    }

    private Validator<String, Defect> getUserValidator(@Nullable User user) {
        return login -> {
            if (user == null) {
                return ValidationResult.failed(login, CommonDefects.objectNotFound());
            }
            if (user.getRole() == null || !user.getRole().isInternal()) {
                return ValidationResult.failed(login, CommonDefects.inconsistentState());
            }
            return ValidationResult.success(login);
        };
    }

    private Validator<Long, Defect> getGroupValidator(@Nullable IdmGroup group) {
        return idmGroupId ->
        {
            if (group == null) {
                return ValidationResult.failed(idmGroupId, CommonDefects.objectNotFound());
            }
            return ValidationResult.success(idmGroupId);
        };
    }

}
