package ru.yandex.direct.internaltools.tools.idm.validation;

import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.idm.model.IdmGroup;
import ru.yandex.direct.core.entity.idm.model.IdmGroupRole;
import ru.yandex.direct.core.entity.idm.repository.IdmGroupsRepository;
import ru.yandex.direct.core.entity.idm.service.IdmGroupsRolesService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.internaltools.tools.idm.model.IdmGroupRoleParameters;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.singleton;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Service
public class IdmGroupRoleParametersValidationService {

    private static final Set<RbacRole> CLIENT_REQUIRED_ROLES = Set.of(RbacRole.AGENCY, RbacRole.CLIENT);

    private final IdmGroupsRolesService idmGroupsRolesService;
    private final IdmGroupsRepository idmGroupsRepository;
    private final ClientService clientService;

    public IdmGroupRoleParametersValidationService(IdmGroupsRolesService idmGroupsRolesService,
                                                   IdmGroupsRepository idmGroupsRepository,
                                                   ClientService clientService) {
        this.idmGroupsRolesService = idmGroupsRolesService;
        this.idmGroupsRepository = idmGroupsRepository;
        this.clientService = clientService;
    }

    public static boolean isAddOperation(IdmGroupRoleParameters params) {
        return Objects.equals(IdmGroupRoleParameters.ADD_OPERATION, params.getOperation());
    }

    public static boolean isDeleteOperation(IdmGroupRoleParameters params) {
        return Objects.equals(IdmGroupRoleParameters.DELETE_OPERATION, params.getOperation());
    }

    public static boolean isShowOperation(IdmGroupRoleParameters params) {
        return !isAddOperation(params) && !isDeleteOperation(params);
    }

    public ValidationResult<IdmGroupRoleParameters, Defect> validate(IdmGroupRoleParameters params) {
        ItemValidationBuilder<IdmGroupRoleParameters, Defect> vb =
                ItemValidationBuilder.of(params, Defect.class);
        if (isAddOperation(params) || isDeleteOperation(params)) {
            Long idmGroupId = params.getIdmGroupId();
            Long clientId = params.getClientId();
            vb.item(idmGroupId, IdmGroupRoleParameters.ID_PROP_NAME)
                    .check(notNull())
                    .check(validId(), When.isValid())
                    .checkByFunction(groupId -> isRoleExist(groupId) ? null : CommonDefects.objectNotFound(), When.isValid());
            vb.item(clientId, IdmGroupRoleParameters.CLIENT_ID_PROP_NAME)
                    .check(notNull())
                    .check(validId(), When.isValid())
                    .checkBy(addOperationValidator());
            if (!vb.getResult().hasAnyErrors()) {
                Optional<IdmGroupRole> actualRole =
                        idmGroupsRolesService.getRole(ClientId.fromLong(clientId), idmGroupId);
                if (isAddOperation(params) && actualRole.isPresent()) {
                    vb.getResult().addError(CommonDefects.inconsistentStateAlreadyExists());
                }
                if (isDeleteOperation(params) && actualRole.isEmpty()) {
                    vb.getResult().addError(CommonDefects.objectNotFound());
                }
            }
        }
        return vb.getResult();
    }

    private boolean isRoleExist(Long idxGroupId) {
        List<IdmGroup> groupsRoles = idmGroupsRepository.getGroups(singleton(idxGroupId));
        return !groupsRoles.isEmpty();
    }

    private Validator<? extends Long, Defect> addOperationValidator() {
        return clientId -> {
            if (clientId == null) {
                return ValidationResult.success(clientId);
            }
            Client client = clientService.getClient(ClientId.fromLong(clientId));
            if (client == null) {
                return ValidationResult.failed(clientId, CommonDefects.objectNotFound());
            }
            if (client.getRole() == null || !CLIENT_REQUIRED_ROLES.contains(client.getRole())) {
                return ValidationResult.failed(clientId, CommonDefects.inconsistentState());
            }
            return ValidationResult.success(clientId);
        };
    }

}
