package ru.yandex.direct.internaltools.tools.mediascope;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.banner.service.MediascopePositionService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardedData;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.mediascope.model.MediascopePositionsSendParameters;
import ru.yandex.direct.internaltools.tools.mediascope.model.MediascopePositionsSendResult;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.getLongIdsFromString;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.isStringWithValidLongIds;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Tool(
        name = "Переотправка баннеров в Mediascope",
        label = "mediascope_positions_send",
        description = "Отправляет баннеры в Mediascope, минуя все проверки и условия",
        consumes = MediascopePositionsSendParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.EXECUTE)
@Category(InternalToolCategory.OTHER)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.SUPERREADER, InternalToolAccessRole.DEVELOPER})
public class MediascopePositionsSendTool extends
        MassInternalTool<MediascopePositionsSendParameters, MediascopePositionsSendResult> {

    @Autowired
    MediascopePositionService mediascopePositionService;

    @Autowired
    ShardHelper shardHelper;

    @Override
    protected List<MediascopePositionsSendResult> getMassData(MediascopePositionsSendParameters parameter) {
        Set<Long> bids = getLongIdsFromString(parameter.getBids());

        List<MediascopePositionsSendResult> results = new ArrayList<>();
        ShardedData<Long> bidsByShard = shardHelper.groupByShard(bids, ShardKey.BID);
        bidsByShard.forEach(
                (shard, bidsOnShard) -> {
                    var bidsToPositionsMap = mediascopePositionService.collectPositions(shard, bidsOnShard);
                    var errorsToPositionsMap = mediascopePositionService.sendPositions(bidsToPositionsMap.values());

                    for (var bid : bidsOnShard) {
                        String result;
                        if (bidsToPositionsMap.containsKey(bid)) {
                            var positionId = bidsToPositionsMap.get(bid).getPositionId();
                            result = errorsToPositionsMap.getOrDefault(positionId, "Ok");
                        } else {
                            result = "Couldn't collect data for banner";
                        }

                        results.add(new MediascopePositionsSendResult(bid, result));
                    }
                }
        );

        var bidsWithShard = listToSet(results, MediascopePositionsSendResult::getBid);
        var invalidBidsResult = bids.stream()
                .filter(bid -> !bidsWithShard.contains(bid))
                .map(bid -> new MediascopePositionsSendResult(bid, "Invalid bannerId"))
                .collect(toList());
        results.addAll(invalidBidsResult);

        return results;
    }

    @Override
    public ValidationResult<MediascopePositionsSendParameters, Defect> validate(
            MediascopePositionsSendParameters mediascopePositionsSendParameters
    ) {
        ItemValidationBuilder<MediascopePositionsSendParameters, Defect> validationBuilder =
                ItemValidationBuilder.of(mediascopePositionsSendParameters);
        validationBuilder
                .item(mediascopePositionsSendParameters.getBids(), "bids")
                .check(isStringWithValidLongIds(1));
        return validationBuilder.getResult();
    }
}
