package ru.yandex.direct.internaltools.tools.minpayab;

import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import one.util.streamex.StreamEx;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.tools.minpayab.model.ABParameter;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.JsonUtils.toJson;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

public abstract class BaseABTool implements BaseInternalTool<ABParameter> {

    private static final String LINE_REGEXP = "\\s*+-\\s*+";

    private final PpcProperty<String> property;

    BaseABTool(PpcPropertiesSupport ppcPropertiesSupport) {
        this.property = ppcPropertiesSupport.get(getProperty());
    }

    /**
     * получить проперти, в которой храним распределение
     */
    abstract PpcPropertyName<String> getProperty();

    /**
     * получить регулярное выражение на матчинг строчки инпута
     */
    abstract String getMatchingRegexp();

    /**
     * получить описание одного распределения
     */
    abstract String objDesc(DistObject obj);

    /**
     * получить компаратор для сортировки распределений для записи в базу
     */
    abstract Comparator<String[]> sortingComparator();

    @Override
    public ValidationResult<ABParameter, Defect> validate(ABParameter abParameter) {

        if (abParameter.getDelete()) {
            return ValidationResult.success(abParameter);
        }

        String regexp = getMatchingRegexp();

        ItemValidationBuilder<ABParameter, Defect> vb = ItemValidationBuilder.of(abParameter);

        vb.item(abParameter.getInput(), "заданное распределение")
                .check(CommonConstraints.notNull(), When.isFalse(abParameter.getDelete()))
                .check(fromPredicate(s -> s.lines()
                                .map(line -> line.matches(regexp))
                                .reduce(Boolean::logicalAnd)
                                .orElse(Boolean.FALSE),
                        CommonDefects.invalidFormat()),
                        When.isValid())
                .check(fromPredicate(s -> {
                            // проверяем сумму процентов по всем вариантам
                            List<Integer> percents = s.lines()
                                    .map(line -> line.split(LINE_REGEXP)[1])
                                    .map(Integer::parseInt)
                                    .collect(Collectors.toList());
                            return percents.stream().noneMatch(Integer.valueOf(0)::equals)
                                    && percents.stream().reduce(Integer::sum).orElse(0).equals(100);
                        },
                        CommonDefects.invalidValue()),
                        When.isValid());
        return vb.getResult();
    }

    @Override
    public InternalToolResult process(ABParameter abParameter) {
        if (abParameter.getDelete()) {
            property.remove();
            return generateResult("Удалено.");
        } else {
            List<DistObject> objList = abParameter.getInput()
                    .lines()
                    .map(s -> s.split(LINE_REGEXP))
                    .sorted(sortingComparator())
                    .map(a -> new DistObject(a[0], Integer.valueOf(a[1])))
                    .collect(Collectors.toList());
            property.set(toJson(objList));
            return generateResult("Изменено");
        }
    }

    @Override
    public InternalToolResult processWithoutInput() {
        return generateResult(property.get() != null ?
                "ВНИМАНИЕ! Эксперимент уже запущен. Изменение процентов исказит результаты. " : "");
    }

    private InternalToolResult generateResult(String body) {
        String result;
        String currentPropValue = property.get();
        if (currentPropValue != null) {
            result = StreamEx.of(fromJson(currentPropValue).elements())
                    .map(e -> fromJson(e, DistObject.class))
                    .map(this::objDesc)
                    .collect(Collectors.joining("; "));
        } else {
            result = "не задано";
        }
        return new InternalToolResult(body + " Текущее значение: " + result);
    }

    @JsonInclude(JsonInclude.Include.NON_NULL)
    static class DistObject {
        @JsonProperty
        private String sum;

        @JsonProperty
        private Integer percent;

        DistObject() {
        }

        DistObject(String sum, Integer percent) {
            this.sum = sum;
            this.percent = percent;
        }

        String getSum() {
            return sum;
        }

        Integer getPercent() {
            return percent;
        }
    }
}
