package ru.yandex.direct.internaltools.tools.modexport;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.moderation.service.ModerationExportQueueService;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Disclaimers;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.modexport.model.UpdateParIdModExportQueueParameters;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.internaltools.tools.modexport.model.UpdateParIdModExportQueueParameters.CAMPAIGN_IDS_FIELD_NAME;
import static ru.yandex.direct.internaltools.tools.modexport.model.UpdateParIdModExportQueueParameters.NEED_DELETE_FIELD_NAME;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.getLongIdsFromString;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.isStringWithValidLongIds;
import static ru.yandex.direct.validation.constraint.CommonConstraints.unconditional;
import static ru.yandex.direct.validation.defect.CommonDefects.mustBeEmpty;

@Tool(
        name = "Назначение par_id в mod_export_queue",
        label = "update_par_id_in_mod_export_queue",
        description = "Позволяет переназначить кампаниям номер процесса par_id в очереди заказов на модерацию",
        consumes = UpdateParIdModExportQueueParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.MODERATE)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.PLACER})
@Disclaimers({
        "Используй этот отчет, только если знаешь что делаешь и к каким последствиям эти изменения могут привести"
})
@ParametersAreNonnullByDefault
public class UpdateParIdModExportQueueTool implements BaseInternalTool<UpdateParIdModExportQueueParameters> {

    private final ModerationExportQueueService moderationExportQueueService;

    @Autowired
    public UpdateParIdModExportQueueTool(ModerationExportQueueService moderationExportQueueService) {
        this.moderationExportQueueService = moderationExportQueueService;
    }

    @Override
    public InternalToolResult process(UpdateParIdModExportQueueParameters parameters) {
        Set<Long> campaignIds = getLongIdsFromString(parameters.getCampaignIds());
        Set<Long> queuedCampaignIds = moderationExportQueueService.remainQueuedCampaignIds(campaignIds);
        Map<Long, Long> parIdByCampaignId = new HashMap<>();
        queuedCampaignIds.forEach(c -> {
            Long parId = parameters.getNeedDelete() ? null : parameters.getParId();
            parIdByCampaignId.put(c, parId);
        });
        moderationExportQueueService.updateParIdByCampaignId(parIdByCampaignId);
        Set<Long> notUpdatedCampaignIds = Sets.difference(campaignIds, queuedCampaignIds);
        String msg = notUpdatedCampaignIds.isEmpty()
                ? "OK"
                : "par_id переназначены для заданных кампаний, кроме отсутствующих в очереди: " + notUpdatedCampaignIds;
        return new InternalToolResult().withMessage(msg);
    }

    @Override
    public ValidationResult<UpdateParIdModExportQueueParameters, Defect> validate(
            UpdateParIdModExportQueueParameters params
    ) {
        ItemValidationBuilder<UpdateParIdModExportQueueParameters, Defect> vb = ItemValidationBuilder.of(params);
        vb.item(params.getNeedDelete(), NEED_DELETE_FIELD_NAME)
                .check(unconditional(mustBeEmpty()), When.isTrue(params.getParId() != null && params.getNeedDelete()));
        vb.item(params.getCampaignIds(), CAMPAIGN_IDS_FIELD_NAME)
                .check(isStringWithValidLongIds(1));
        return vb.getResult();
    }

}
