package ru.yandex.direct.internaltools.tools.modreasons;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.moderationreason.repository.ModerationReasonRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.modreasons.model.DeleteModReasonsParameters;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.internaltools.tools.modreasons.model.DeleteModReasonsParameters.CID_FIELD_NAME;
import static ru.yandex.direct.internaltools.tools.modreasons.model.DeleteModReasonsParameters.CLIENTID_FIELD_NAME;
import static ru.yandex.direct.internaltools.tools.modreasons.model.DeleteModReasonsParameters.SHARD_FIELD_NAME;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.validation.builder.When.isValid;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.NumberConstraints.greaterThan;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Tool(
        name = "Удаление записей из mod_reasons",
        label = "_delete_mod_reasons_tool",
        description = "Удаляет в шарде строки из mod_reasons, соответствующие ClientID и cid\n" +
                "Строки должны быть со statusSending != Yes и не иметь парных записей в mod_mail_candidates\n" +
                "Может вычислять шард самостоятельно по ClientID",
        consumes = DeleteModReasonsParameters.class,
        type = InternalToolType.WRITER
)
@Category(InternalToolCategory.MODERATE)
@Action(InternalToolAction.DELETE)
@AccessGroup({InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPER, InternalToolAccessRole.PLACER})
@ParametersAreNonnullByDefault
public class DeleteModReasonsTool implements BaseInternalTool<DeleteModReasonsParameters> {
    private static final Logger logger = LoggerFactory.getLogger(DeleteModReasonsTool.class);
    private final ModerationReasonRepository moderationReasonRepository;
    private final ShardHelper shardHelper;

    @Autowired
    public DeleteModReasonsTool(ModerationReasonRepository moderationReasonRepository,
                                ShardHelper shardHelper) {
        this.moderationReasonRepository = moderationReasonRepository;
        this.shardHelper = shardHelper;
    }

    @Override
    public ValidationResult<DeleteModReasonsParameters, Defect> validate(DeleteModReasonsParameters parameters) {
        ItemValidationBuilder<DeleteModReasonsParameters, Defect> validationBuilder =
                ItemValidationBuilder.of(parameters);
        validationBuilder.item(parameters.getShard(), SHARD_FIELD_NAME)
                .check(greaterThan(0));
        validationBuilder.item(parameters.getCid(), CID_FIELD_NAME)
                .check(validId());
        validationBuilder.item(parameters.getClientId(), CLIENTID_FIELD_NAME)
                .check(validId());

        validationBuilder.check(Constraint.fromPredicate(this::shardCanBeObtained, objectNotFound()), isValid());

        return validationBuilder.getResult();
    }

    public InternalToolResult process(DeleteModReasonsParameters parameters) {
        ClientId clientId = ClientId.fromLong(parameters.getClientId());
        int shard = guessShard(parameters);
        logger.debug("Удаляем в шарде {} запись для клиента {} и кампании {}", shard, clientId, parameters.getCid());
        int recordsDeleted =
                moderationReasonRepository.deleteUnsentModerationReasons(shard, clientId, parameters.getCid());
        logger.debug("Удалено {} строк из шарда {}", recordsDeleted, shard);
        return new InternalToolResult().withMessage("Удалено " + recordsDeleted + " строк.");
    }

    private int guessShard(DeleteModReasonsParameters parameters) {
        return nvl(parameters.getShard(), shardHelper.getShardByClientId(ClientId.fromLong(parameters.getClientId())));
    }

    private boolean shardCanBeObtained(DeleteModReasonsParameters parameters) {
        return guessShard(parameters) > 0;
    }
}
