package ru.yandex.direct.internaltools.tools.multicurrency.container;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.text.DecimalFormat;

import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.direct.currency.ConstantDescription;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.currency.CurrencyCode.BYN;
import static ru.yandex.direct.currency.CurrencyCode.CHF;
import static ru.yandex.direct.currency.CurrencyCode.EUR;
import static ru.yandex.direct.currency.CurrencyCode.GBP;
import static ru.yandex.direct.currency.CurrencyCode.KZT;
import static ru.yandex.direct.currency.CurrencyCode.RUB;
import static ru.yandex.direct.currency.CurrencyCode.TRY;
import static ru.yandex.direct.currency.CurrencyCode.UAH;
import static ru.yandex.direct.currency.CurrencyCode.USD;
import static ru.yandex.direct.currency.CurrencyCode.YND_FIXED;

/**
 * Информация о значении валютной константы для всех валют.
 */
public class CurrenciesConstantInfo {
    private static final String EMPTY = "—";

    private Method currencyMethod;

    /**
     * Получить описание и значения константы для разных валют
     *
     * @param method метод интерфейса {@link Currency}
     * @throws IllegalArgumentException если метод не принадлежит классу {@link Currency}
     */
    public CurrenciesConstantInfo(Method method) {
        checkArgument(method.getDeclaringClass().equals(Currency.class));
        currencyMethod = method;
    }

    @JsonProperty(value = "Константа")
    public String getDescription() {
        ConstantDescription description = currencyMethod.getDeclaredAnnotation(ConstantDescription.class);
        return description != null ? description.value() : EMPTY;
    }

    @JsonProperty(value = "у.е.")
    public Object getValueForYndFixed() {
        return getValue(YND_FIXED);
    }

    @JsonProperty(value = "руб")
    public Object getValueForRub() {
        return getValue(RUB);
    }

    @JsonProperty(value = "грн")
    public Object getValueForUah() {
        return getValue(UAH);
    }

    @JsonProperty(value = "долл.")
    public Object getValueForUsd() {
        return getValue(USD);
    }

    @JsonProperty(value = "евро")
    public Object getValueForEur() {
        return getValue(EUR);
    }

    @JsonProperty(value = "BYN")
    public Object getValueForByn() {
        return getValue(BYN);
    }

    @JsonProperty(value = "тенге")
    public Object getValueForKzt() {
        return getValue(KZT);
    }

    @JsonProperty(value = "CHF")
    public Object getValueForChf() {
        return getValue(CHF);
    }

    @JsonProperty(value = "TL")
    public Object getValueForTry() {
        return getValue(TRY);
    }

    @JsonProperty(value = "GBP")
    public Object getValueForGbp() {
        return getValue(GBP);
    }

    private Object getValue(CurrencyCode currencyCode) {
        try {
            Object value = currencyMethod.invoke(Currencies.getCurrency(currencyCode));
            if (value == null) {
                return EMPTY;
            } else if (value instanceof Number) {
                return DecimalFormat.getInstance().format(value);
            } else {
                return value;
            }
        } catch (IllegalAccessException | InvocationTargetException e) {
            return EMPTY;
        }
    }
}
