package ru.yandex.direct.internaltools.tools.newinterface;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.YesNo;
import ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.newinterface.container.NewInterfaceAlertValue;
import ru.yandex.direct.internaltools.tools.newinterface.model.NewInterfaceAlertParameters;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.unconditional;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

@Tool(
        name = "Управление плашкой-предупреждением в новом интерфейсе",
        label = "new_interface_alert",
        description = "- Выключить: \n" +
                "  1. отправь форму с выключенной галочкой\n" +
                "  2. проверь в интерфейсе\n" +
                "- Включить:\n" +
                "  1. включи галочку, заполни русский и английский тексты\n" +
                "  2. при необходимости укажи список ClientID. Плашка будет видна только им\n" +
                "  3. проверь в интерфейсе\n" +
                "\n" +
                "ИСПОЛЬЗОВАТЬ ТОЛЬКО С ПОЛНЫМ ПОНИМАНИЕМ ТОГО, ЧТО ПРОИЗОЙДЕТ!\n" +
                "ЕСТЬ ВОЗМОЖНОСТЬ ПОВЛИЯТЬ НА ИНТЕРФЕЙС ВСЕХ КЛИЕНТОВ!",
        consumes = NewInterfaceAlertParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.SET)
@Category(InternalToolCategory.NEW_INTERFACE)
@AccessGroup({InternalToolAccessRole.SUPER})
@ParametersAreNonnullByDefault
public class NewInterfaceAlertTool extends MassInternalTool<NewInterfaceAlertParameters, NewInterfaceAlertValue> {
    public static final String ALERT_TEXT_RU_FIELD = "русский текст";
    public static final String ALERT_TEXT_EN_FIELD = "английский текст";
    public static final String CLIENT_IDS_FIELD = "список ClientID";
    private static final Pattern WORD_DELIMITER = Pattern.compile("\\W+", Pattern.UNICODE_CHARACTER_CLASS);

    private final PpcPropertiesSupport propertiesSupport;

    @Autowired
    public NewInterfaceAlertTool(PpcPropertiesSupport propertiesSupport) {
        this.propertiesSupport = propertiesSupport;
    }

    @Override
    protected List<NewInterfaceAlertValue> getMassData(NewInterfaceAlertParameters parameter) {
        propertiesSupport.set(PpcPropertyEnum.NEW_INTERFACE_ALERT_ENABLED.getName(),
                parameter.getEnableAlert() ? YesNo.YES.name() : YesNo.NO.name());
        propertiesSupport.set(PpcPropertyEnum.NEW_INTERFACE_ALERT_TEXT_RU.getName(), parameter.getAlertTextRu());
        propertiesSupport.set(PpcPropertyEnum.NEW_INTERFACE_ALERT_TEXT_EN.getName(), parameter.getAlertTextEn());
        String clientIds = sanitizeLongListString(parameter.getClientIds());
        propertiesSupport.set(PpcPropertyEnum.NEW_INTERFACE_ALERT_TEXT_CLIENT_IDS.getName(), clientIds);
        return getMassData();
    }

    @Override
    protected List<NewInterfaceAlertValue> getMassData() {
        Map<String, String> alertProperties = propertiesSupport.getByNames(
                Arrays.asList(PpcPropertyEnum.NEW_INTERFACE_ALERT_ENABLED.getName(),
                        PpcPropertyEnum.NEW_INTERFACE_ALERT_TEXT_EN.getName(),
                        PpcPropertyEnum.NEW_INTERFACE_ALERT_TEXT_RU.getName(),
                        PpcPropertyEnum.NEW_INTERFACE_ALERT_TEXT_CLIENT_IDS.getName()));
        return Collections.singletonList(
                new NewInterfaceAlertValue(alertProperties.get(PpcPropertyEnum.NEW_INTERFACE_ALERT_ENABLED.getName()),
                        alertProperties.get(PpcPropertyEnum.NEW_INTERFACE_ALERT_TEXT_RU.getName()),
                        alertProperties.get(PpcPropertyEnum.NEW_INTERFACE_ALERT_TEXT_EN.getName()),
                        alertProperties.get(PpcPropertyEnum.NEW_INTERFACE_ALERT_TEXT_CLIENT_IDS.getName())));
    }

    @Override
    public ValidationResult<NewInterfaceAlertParameters, Defect> validate(
            NewInterfaceAlertParameters parameters) {
        ItemValidationBuilder<NewInterfaceAlertParameters, Defect> validationBuilder =
                ItemValidationBuilder.of(parameters);

        validationBuilder
                .item(parameters.getAlertTextRu(), ALERT_TEXT_RU_FIELD)
                .check(notNull(), When.isTrue(parameters.getEnableAlert()))
                .check(notBlank(), When.isTrue(parameters.getEnableAlert()));
        validationBuilder
                .item(parameters.getAlertTextEn(), ALERT_TEXT_EN_FIELD)
                .check(notNull(), When.isTrue(parameters.getEnableAlert()))
                .check(notBlank(), When.isTrue(parameters.getEnableAlert()));
        validationBuilder
                .item(parameters.getClientIds(), CLIENT_IDS_FIELD)
                .checkBy(clientIdsListValidator());
        return validationBuilder.getResult();
    }

    @SuppressWarnings("rawtypes")
    private static Validator<String, Defect> clientIdsListValidator() {
        return clientIdsStr -> {
            ItemValidationBuilder<String, Defect> ivb = ItemValidationBuilder.of(clientIdsStr);
            if (clientIdsStr == null) {
                return ivb.getResult();
            }

            ivb.check(unconditional(invalidValue()), When.isFalse(isValidClientIdList(clientIdsStr)));

            return ivb.getResult();
        };
    }

    private static boolean isValidClientIdList(String clientIdsStr) {
        try {
            sanitizeLongListString(clientIdsStr);
            return true;
        } catch (RuntimeException e) {
            return false;
        }
    }

    /**
     * @return строка чисел, разделённых запятыми
     * @throws NumberFormatException если в строке есть не только числа
     */
    static String sanitizeLongListString(String longListString) {
        return StreamEx.split(longListString, WORD_DELIMITER)
                .remove(String::isEmpty)
                .map(Long::parseLong)
                .joining(",");
    }

}
