package ru.yandex.direct.internaltools.tools.oneshot.launch;

import java.util.Arrays;
import java.util.Objects;
import java.util.function.Supplier;

import ru.yandex.direct.env.Environment;
import ru.yandex.direct.internaltools.tools.oneshot.launch.model.OneshotLaunchAction;
import ru.yandex.direct.oneshot.core.model.Oneshot;
import ru.yandex.direct.oneshot.core.model.OneshotLaunch;
import ru.yandex.direct.oneshot.core.model.OneshotLaunchValidationStatus;

@SuppressWarnings("unchecked")
public class OneshotLaunchActionValidator {
    private OneshotLaunchAction oneshotLaunchAction;
    private Oneshot oneshot;
    private OneshotLaunch oneshotLaunch;
    private String operatorLogin;

    public OneshotLaunchActionValidator(OneshotLaunchAction oneshotLaunchAction,
                                        Oneshot oneshot, OneshotLaunch oneshotLaunch, String operatorLogin) {
        this.oneshotLaunchAction = oneshotLaunchAction;
        this.oneshot = oneshot;
        this.oneshotLaunch = oneshotLaunch;
        this.operatorLogin = operatorLogin;
    }

    /**
     * Возвращает сообщение об ошибке или {@code null}, если все хорошо
     */
    public String validate() {
        switch (oneshotLaunchAction) {
            case APPROVE:
                return validateApprove();
            case LAUNCH:
                return validateLaunch();
            case PAUSE:
                return validatePause();
            case CANCEL:
                return validateCancel();
            case DELETE:
                return validateDelete();
            default:
                throw new UnsupportedOperationException("Unknown action");
        }
    }

    private String validateApprove() {
        // Не проверяем оператора в непродакшен окружениях
        boolean isNonProduction = !Environment.getCached().isProductionOrPrestable();
        return checkAll(
                check(() -> !isLaunchApproved(), "Запуск уже подтвержден"),
                check(() -> oneshotLaunch.getValidationStatus() == OneshotLaunchValidationStatus.VALID,
                        "Запуск не валиден/еще не провалидирован"),
                check(() -> isNonProduction || !operatorLogin.equals(oneshotLaunch.getLaunchCreator()),
                        "Заказчик не может подтверждать собственные запуски"),
                check(() -> isNonProduction || isApprover(), "Вы не в списке разрешенных апруверов")
        );
    }

    private String validateLaunch() {
        return checkAll(
                checkNotLaunched(),
                checkNonProductionOrSafeOneshotOrLaunchApproved(),
                checkOperatorIsCreatorOrApprover()
        );
    }

    private String validateDelete() {
        return checkAll(
                checkNotLaunched(),
                checkOperatorIsCreatorOrApproverOrSafeOneshot()
        );
    }

    private String validatePause() {
        return checkAll(
                checkLaunched()
        );
    }

    private String validateCancel() {
        return checkAll(
                checkLaunched(),
                checkOperatorIsCreatorOrApproverOrSafeOneshot()
        );
    }

    private Supplier<String> checkNotLaunched() {
        return check(() -> oneshotLaunch.getLaunchRequestTime() == null,
                "Действие невозможно. Запуск уже совершен (попробуйте отмену/паузу)");
    }

    private Supplier<String> checkLaunched() {
        return check(() -> oneshotLaunch.getLaunchRequestTime() != null, "Ваншот еще не запущен");
    }

    private boolean isApprover() {
        return oneshot.getApprovers().contains(operatorLogin);
    }

    private boolean isLaunchApproved() {
        return oneshotLaunch.getApprover() != null;
    }

    private Supplier<String> checkOperatorIsCreatorOrApproverOrSafeOneshot() {
        // Не проверяем оператора в непродакшен окружениях
        boolean isNonProduction = !Environment.getCached().isProductionOrPrestable();
        return check(() -> isNonProduction || operatorLogin.equals(oneshotLaunch.getLaunchCreator())
                        || isApprover() || oneshot.getSafeOneshot(),
                "Действие может выполнить только создатель или один из апруверов");
    }

    private Supplier<String> checkNonProductionOrSafeOneshotOrLaunchApproved() {
        // Не проверяем оператора в непродакшен окружениях
        boolean isNonProduction = !Environment.getCached().isProductionOrPrestable();
        return check(() -> isNonProduction || oneshot.getSafeOneshot() || isLaunchApproved(),
                "Требуется аппрув");
    }

    private Supplier<String> checkOperatorIsCreatorOrApprover() {
        // Не проверяем оператора в непродакшен окружениях
        return check(() -> operatorLogin.equals(oneshotLaunch.getLaunchCreator()) || isApprover(),
                "Действие может выполнить только создатель или один из апруверов");
    }

    /**
     * Если условие выполняется, возвращает null, иначе заданное сообщение об ошибке
     *
     * @param condition    - supplier, проверяющий условие
     * @param errorMessage - сообщение об ошибке
     */
    private Supplier<String> check(Supplier<Boolean> condition, String errorMessage) {
        return () -> condition.get() ? null : errorMessage;
    }

    /**
     * Последовательно применяет проверки, возвращает результат первой проваленной
     *
     * @param checkers - проверки
     * @return {@code null} - если проверки успешно пройдены, или текст первой возникшей ошибки
     */
    private String checkAll(Supplier<String>... checkers) {
        return Arrays.stream(checkers)
                .map(Supplier::get)
                .filter(Objects::nonNull)
                .findFirst()
                .orElse(null);
    }
}
