package ru.yandex.direct.internaltools.tools.oneshot.launch;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;

import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolMassResult;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.oneshot.OneshotToolsUtil;
import ru.yandex.direct.internaltools.tools.oneshot.launch.model.OneshotLaunchInfo;
import ru.yandex.direct.internaltools.tools.oneshot.launch.model.OneshotLaunchInput;
import ru.yandex.direct.internaltools.tools.oneshot.launchdata.OneshotLaunchDataProcessor;
import ru.yandex.direct.oneshot.core.entity.oneshot.OneshotAppVersionProvider;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotLaunchDataRepository;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotLaunchRepository;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotRepository;
import ru.yandex.direct.oneshot.core.entity.oneshot.service.OneshotStartrekService;
import ru.yandex.direct.oneshot.core.model.LaunchStatus;
import ru.yandex.direct.oneshot.core.model.Oneshot;
import ru.yandex.direct.oneshot.core.model.OneshotLaunch;
import ru.yandex.direct.oneshot.core.model.OneshotLaunchData;

import static java.util.stream.Collectors.groupingBy;
import static ru.yandex.direct.internaltools.tools.oneshot.LogviewerUrlUtil.getLaunchLogUrl;
import static ru.yandex.direct.internaltools.tools.oneshot.OneshotToolsUtil.SAFE_ONESHOT_APPROVER;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Tool(
        name = "Запуски ваншотов",
        label = "oneshot_launches",
        description = "Флоу запуска: создание -> валидация -> апрув -> " +
                "запуск потока выполнения (или нескольких потоков для шардированных ваншотов) " +
                "(вручную, в любой момент после апрува)",
        consumes = OneshotLaunchInput.class,
        type = InternalToolType.WRITER
)
@Category(InternalToolCategory.ONESHOTS)
@Action(InternalToolAction.EXECUTE)
@AccessGroup({InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPER})
@ParametersAreNonnullByDefault
public class OneshotLaunchesTool implements BaseInternalTool<OneshotLaunchInput> {

    public static final Comparator<OneshotLaunch> SORTING_COMPARATOR =
            Comparator.comparing(OneshotLaunch::getId).reversed();

    private OneshotRepository oneshotRepository;
    private OneshotLaunchRepository oneshotLaunchRepository;
    private OneshotLaunchDataRepository oneshotLaunchDataRepository;
    private OneshotStartrekService oneshotStartrekService;
    private DslContextProvider dslContextProvider;
    private ShardHelper shardHelper;
    private OneshotAppVersionProvider oneshotAppVersionProvider;

    public OneshotLaunchesTool(OneshotRepository oneshotRepository, OneshotLaunchRepository oneshotLaunchRepository,
                               OneshotLaunchDataRepository oneshotLaunchDataRepository,
                               OneshotStartrekService oneshotStartrekService,
                               DslContextProvider dslContextProvider, ShardHelper shardHelper,
                               OneshotAppVersionProvider oneshotAppVersionProvider) {
        this.oneshotRepository = oneshotRepository;
        this.oneshotLaunchRepository = oneshotLaunchRepository;
        this.oneshotLaunchDataRepository = oneshotLaunchDataRepository;
        this.oneshotStartrekService = oneshotStartrekService;
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.oneshotAppVersionProvider = oneshotAppVersionProvider;
    }

    /**
     * Получаем информацию о запусках: имя класса, статусы, ссылка на логи и прочее.
     */
    @Override
    public InternalToolResult processWithoutInput() {
        Map<Long, Oneshot> oneshotById = oneshotRepository.getExistingById();
        List<OneshotLaunch> oneshotLaunches = oneshotLaunchRepository.getByOneshotIds(oneshotById.keySet());
        Set<Long> launchIds = listToSet(oneshotLaunches, OneshotLaunch::getId);
        Map<Long, List<OneshotLaunchData>> oneshotLaunchData =
                oneshotLaunchDataRepository.getLaunchDataByLaunchId(launchIds);
        List<OneshotLaunchInfo> result = createOneshotLaunchInfoList(oneshotById, oneshotLaunches, oneshotLaunchData);
        return new InternalToolMassResult<>(result);
    }

    private List<OneshotLaunchInfo> createOneshotLaunchInfoList(Map<Long, Oneshot> oneshotById,
                                                                List<OneshotLaunch> oneshotLaunches,
                                                                Map<Long, List<OneshotLaunchData>> oneshotLaunchData) {
        return StreamEx.of(oneshotLaunches)
                .sorted(SORTING_COMPARATOR)
                .map(launch -> {
                    Oneshot oneshot = oneshotById.get(launch.getOneshotId());
                    List<OneshotLaunchData> oneshotLaunchDataList = oneshotLaunchData.get(launch.getId());
                    return createOneshotLaunchInfo(oneshot, launch, oneshotLaunchDataList);
                })
                .toList();
    }

    private OneshotLaunchInfo createOneshotLaunchInfo(Oneshot oneshot, OneshotLaunch launch,
                                                      List<OneshotLaunchData> oneshotLaunchData) {
        return new OneshotLaunchInfo()
                .withLaunchId(launch.getId())
                .withClassName(OneshotToolsUtil.cutOneshotClass(oneshot.getClassName()))
                .withLaunchCreator(launch.getLaunchCreator())
                .withParams(launch.getParams())
                .withValidationStatus(launch.getValidationStatus())
                .withApprover(getApprover(oneshot, launch))
                .withApprovedRevision(launch.getApprovedRevision())
                .withLaunchRequestTime(launch.getLaunchRequestTime())
                .withOneshotFlowsState(aggregateLaunchStatus(oneshotLaunchData))
                .withLogviewerLink(getLaunchLogUrl(oneshot.getCreateTime(), launch.getTraceId()));
    }

    private String getApprover(Oneshot oneshot, OneshotLaunch launch) {
        if (launch.getApprover() == null) {
            return oneshot.getSafeOneshot() ? SAFE_ONESHOT_APPROVER : null;
        } else {
            return launch.getApprover();
        }
    }

    /**
     * Собираем и возвращаем информацию о статусах потоков запуска в виде строки в формате "статус - количество"
     */
    private String aggregateLaunchStatus(@Nullable List<OneshotLaunchData> launchStatuses) {
        if (launchStatuses == null) {
            return "";
        }
        Map<LaunchStatus, Long> statusesCount = launchStatuses.stream()
                .map(OneshotLaunchData::getLaunchStatus)
                .collect(groupingBy(Function.identity(), Collectors.counting()));
        return EntryStream.of(statusesCount)
                .flatMapKeyValue((status, count) -> Stream.of(status.name() + " - " + count))
                .joining(",");
    }

    @Override
    public InternalToolResult process(OneshotLaunchInput launchInput) {
        dslContextProvider.ppcdictTransaction(conf -> createProcessor(launchInput, conf.dsl()).validateAndProcess());
        return processWithoutInput();
    }

    private OneshotLaunchDataProcessor createProcessor(OneshotLaunchInput launchInput, DSLContext dslContext) {
        return new OneshotLaunchDataProcessor(launchInput, oneshotRepository, oneshotLaunchRepository,
                oneshotLaunchDataRepository, oneshotAppVersionProvider, oneshotStartrekService, shardHelper,
                dslContext);
    }
}
