package ru.yandex.direct.internaltools.tools.oneshot.launchdata;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableMap;
import org.jooq.DSLContext;

import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.tools.oneshot.launch.OneshotLaunchActionValidator;
import ru.yandex.direct.internaltools.tools.oneshot.launch.OneshotLaunchesTool;
import ru.yandex.direct.internaltools.tools.oneshot.launch.model.OneshotLaunchAction;
import ru.yandex.direct.internaltools.tools.oneshot.launch.model.OneshotLaunchInput;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.oneshot.core.entity.oneshot.OneshotAppVersionProvider;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotLaunchDataRepository;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotLaunchRepository;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotRepository;
import ru.yandex.direct.oneshot.core.entity.oneshot.service.OneshotStartrekService;
import ru.yandex.direct.oneshot.core.model.LaunchStatus;
import ru.yandex.direct.oneshot.core.model.Oneshot;
import ru.yandex.direct.oneshot.core.model.OneshotLaunch;
import ru.yandex.direct.oneshot.core.model.OneshotLaunchData;
import ru.yandex.direct.tracing.util.TraceUtil;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class OneshotLaunchDataProcessor {
    /**
     * Какие статусы можно преобразовать в заданный. Ключ - статус, в который хотим перевести.
     * Значение - список статусов, из которых в этот статус переводим.
     */
    private static final Map<LaunchStatus, Set<LaunchStatus>> MANUALLY_CHANGING_STATUS_WORKFLOW =
            ImmutableMap.<LaunchStatus, Set<LaunchStatus>>builder()
                    .put(LaunchStatus.PAUSE_REQUESTED, Set.of(LaunchStatus.READY, LaunchStatus.IN_PROGRESS))
                    .put(LaunchStatus.CANCELED, Set.of(LaunchStatus.READY, LaunchStatus.IN_PROGRESS,
                            LaunchStatus.PAUSED))
                    .build();

    private OneshotLaunchRepository oneshotLaunchRepository;
    private OneshotLaunchDataRepository oneshotLaunchDataRepository;
    private OneshotAppVersionProvider oneshotAppVersionProvider;
    private OneshotStartrekService oneshotStartrekService;

    private ShardHelper shardHelper;
    private DSLContext dslContext;
    private LaunchProcessorContext context;

    public OneshotLaunchDataProcessor(OneshotLaunchInput launchInput,
                                      OneshotRepository oneshotRepository,
                                      OneshotLaunchRepository oneshotLaunchRepository,
                                      OneshotLaunchDataRepository oneshotLaunchDataRepository,
                                      OneshotAppVersionProvider oneshotAppVersionProvider,
                                      OneshotStartrekService oneshotStartrekService,
                                      ShardHelper shardHelper,
                                      DSLContext dslContext) {
        this.oneshotLaunchRepository = oneshotLaunchRepository;
        this.oneshotLaunchDataRepository = oneshotLaunchDataRepository;
        this.oneshotAppVersionProvider = oneshotAppVersionProvider;
        this.oneshotStartrekService = oneshotStartrekService;
        this.shardHelper = shardHelper;
        this.dslContext = dslContext;
        this.context = createContext(launchInput, oneshotRepository);
    }

    public void validateAndProcess() {
        validate();
        process();
    }

    private void process() {
        switch (context.getOneshotLaunchAction()) {
            case APPROVE:
                approve();
                break;
            case LAUNCH:
                launch();
                break;
            case DELETE:
                delete();
                break;
            case PAUSE:
                oneshotLaunchDataRepository.updateLaunchDataStatuses(dslContext, context.getOneshotLaunch().getId(),
                        LaunchStatus.PAUSE_REQUESTED,
                        MANUALLY_CHANGING_STATUS_WORKFLOW.get(LaunchStatus.PAUSE_REQUESTED));
                break;
            case CANCEL:
                oneshotLaunchDataRepository.updateLaunchDataStatuses(dslContext, context.getOneshotLaunch().getId(),
                        LaunchStatus.CANCELED,
                        MANUALLY_CHANGING_STATUS_WORKFLOW.get(LaunchStatus.CANCELED));
                break;
            default:
                throw new UnsupportedOperationException("Unknown action");
        }
    }

    private void validate() {
        String result = new OneshotLaunchActionValidator(context.getOneshotLaunchAction(), context.getOneshot(),
                context.getOneshotLaunch(), context.getOperatorLogin()).validate();
        if (result != null) {
            throw new InternalToolValidationException(result);
        }
    }

    private LaunchProcessorContext createContext(OneshotLaunchInput launchInput, OneshotRepository oneshotRepository) {
        Long launchId = extractLaunchId(launchInput);
        OneshotLaunch oneshotLaunch = oneshotLaunchRepository.getForUpdate(dslContext, launchId);
        Oneshot oneshot = oneshotRepository.get(oneshotLaunch.getOneshotId());
        return new LaunchProcessorContext(launchInput.getLaunchAction(),
                launchInput.getOperator().getDomainLogin(), oneshot, oneshotLaunch);
    }

    private Long extractLaunchId(OneshotLaunchInput launchInput) {
        String stringId = launchInput.getLaunchName().split(":")[0];
        return Long.parseLong(stringId);
    }

    /**
     * Помечаем поток как подтвержденный
     */
    private void approve() {
        OneshotLaunch oneshotLaunch = context.getOneshotLaunch();
        AppliedChanges<OneshotLaunch> changes = new ModelChanges<>(oneshotLaunch.getId(), OneshotLaunch.class)
                .process(context.getOperatorLogin(), OneshotLaunch.APPROVER)
                .process(nvl(oneshotAppVersionProvider.getCurrentRevision(), ""),
                        OneshotLaunch.APPROVED_REVISION)
                .applyTo(oneshotLaunch);
        oneshotLaunchRepository.updateLaunch(dslContext, changes);
        oneshotStartrekService.writeLaunchApprovedComment(context.getOneshot().getTicket(),
                context.getOneshotLaunch().getId(), context.getOperatorLogin(),
                OneshotLaunchesTool.class.getAnnotation(Tool.class).label());
    }

    /**
     * Помечаем поток, как готовый к запуску и создаем для него записи с данными (для потоков).
     */
    private void launch() {
        OneshotLaunch oneshotLaunch = context.getOneshotLaunch();
        AppliedChanges<OneshotLaunch> changes = new ModelChanges<>(oneshotLaunch.getId(), OneshotLaunch.class)
                .process(LocalDateTime.now(), OneshotLaunch.LAUNCH_REQUEST_TIME)
                .applyTo(oneshotLaunch);

        List<OneshotLaunchData> launchFlows = createLaunchFlows(oneshotLaunch, context.getOneshot().getSharded());

        oneshotLaunchRepository.updateLaunch(dslContext, changes);
        oneshotLaunchDataRepository.add(dslContext, launchFlows);
        oneshotStartrekService.writeStartLaunchComment(context.getOneshot().getTicket(),
                context.getOneshotLaunch().getId(), context.getOperatorLogin(),
                OneshotLaunchDataTool.class.getAnnotation(Tool.class).label());
    }

    /**
     * Удаляем поток из базы
     */
    private void delete() {
        int deletedCount = oneshotLaunchRepository.deleteLaunch(dslContext, context.getOneshotLaunch().getId());
        if (deletedCount == 0) {
            throw new RuntimeException("Oneshot has already been launched");
        } else {
            oneshotStartrekService.writeDeleteLaunchComment(context.getOneshot().getTicket(),
                    context.getOneshotLaunch().getId(), context.getOperatorLogin());
        }
    }

    /**
     * Создаем объекты потоков запуска.
     *
     * @param oneshotLaunch - запуск, для которого надо создать потоки
     * @param sharded       - шардированный ли ваншот
     * @return объекты потоков запуска (1 - если ваншот не шардированный, или в количестве шардов)
     */
    private List<OneshotLaunchData> createLaunchFlows(OneshotLaunch oneshotLaunch, boolean sharded) {
        return sharded
                ? mapList(shardHelper.dbShards(), shard -> createNewLaunch(oneshotLaunch.getId(), shard))
                : singletonList(createNewLaunch(oneshotLaunch.getId(), 0));
    }

    private OneshotLaunchData createNewLaunch(Long launchId, int shard) {
        return new OneshotLaunchData()
                .withLaunchId(launchId)
                .withLaunchStatus(LaunchStatus.READY)
                .withShard(shard)
                .withSpanId(TraceUtil.randomId());
    }

    private static final class LaunchProcessorContext {
        private OneshotLaunchAction oneshotLaunchAction;
        private String operatorLogin;
        private Oneshot oneshot;
        private OneshotLaunch oneshotLaunch;

        LaunchProcessorContext(OneshotLaunchAction oneshotLaunchAction, String operatorLogin, Oneshot oneshot,
                               OneshotLaunch oneshotLaunch) {
            this.oneshotLaunchAction = oneshotLaunchAction;
            this.operatorLogin = operatorLogin;
            this.oneshot = oneshot;
            this.oneshotLaunch = oneshotLaunch;
        }

        OneshotLaunchAction getOneshotLaunchAction() {
            return oneshotLaunchAction;
        }

        String getOperatorLogin() {
            return operatorLogin;
        }

        Oneshot getOneshot() {
            return oneshot;
        }

        OneshotLaunch getOneshotLaunch() {
            return oneshotLaunch;
        }
    }
}
