package ru.yandex.direct.internaltools.tools.oneshot.launchdata;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.oneshot.launchdata.model.LaunchDataInfo;
import ru.yandex.direct.internaltools.tools.oneshot.launchdata.model.LaunchDataInput;
import ru.yandex.direct.internaltools.utils.OneshotDefects;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotLaunchDataRepository;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotLaunchRepository;
import ru.yandex.direct.oneshot.core.entity.oneshot.repository.OneshotRepository;
import ru.yandex.direct.oneshot.core.model.LaunchStatus;
import ru.yandex.direct.oneshot.core.model.Oneshot;
import ru.yandex.direct.oneshot.core.model.OneshotLaunch;
import ru.yandex.direct.oneshot.core.model.OneshotLaunchData;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.internaltools.tools.oneshot.LogviewerUrlUtil.getLaunchDataLogUrl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Tool(
        name = "Потоки выполнения ваншотов",
        label = "oneshot_launch_data",
        description = "",
        consumes = LaunchDataInput.class,
        type = InternalToolType.WRITER
)
@Category(InternalToolCategory.ONESHOTS)
@Action(InternalToolAction.EXECUTE)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class OneshotLaunchDataTool extends MassInternalTool<LaunchDataInput, LaunchDataInfo> {

    private final OneshotLaunchDataRepository oneshotLaunchDataRepository;
    private final OneshotLaunchRepository oneshotLaunchRepository;
    private final OneshotRepository oneshotRepository;
    private final DslContextProvider dslContextProvider;


    private static final Map<LaunchStatus, Set<LaunchStatus>> VALID_TRANSITIONS = Map.of(
            LaunchStatus.IN_PROGRESS, Set.of(LaunchStatus.PAUSE_REQUESTED, LaunchStatus.CANCELED),
            LaunchStatus.PAUSED, Set.of(LaunchStatus.CANCELED, LaunchStatus.READY),
            LaunchStatus.READY, Set.of(LaunchStatus.PAUSED, LaunchStatus.CANCELED)
    );

    @Autowired
    public OneshotLaunchDataTool(OneshotLaunchDataRepository oneshotLaunchDataRepository,
                                 OneshotLaunchRepository oneshotLaunchRepository,
                                 OneshotRepository oneshotRepository,
                                 DslContextProvider dslContextProvider) {
        this.oneshotLaunchDataRepository = oneshotLaunchDataRepository;
        this.oneshotLaunchRepository = oneshotLaunchRepository;
        this.oneshotRepository = oneshotRepository;
        this.dslContextProvider = dslContextProvider;
    }


    @Override
    protected List<LaunchDataInfo> getMassData(LaunchDataInput parameter) {
        LaunchStatus toStatus = parameter.getAction().getLaunchStatus();
        if (parameter.getLaunchDataId() != null) {
            runByLaunchDataId(parameter.getLaunchDataId(), toStatus, parameter.getOperator());
        } else {
            runByLaunchId(parameter.getLaunchId(), toStatus, parameter.getOperator());
        }
        return getMassData();
    }

    private void runByLaunchDataId(Long launchDataId, LaunchStatus status, User operator) {
        dslContextProvider.ppcdictTransaction(c -> {
            DSLContext dslContext = c.dsl();
            OneshotLaunchData selected = oneshotLaunchDataRepository.selectForUpdate(dslContext,
                    launchDataId);
            if (selected == null) {
                throw new RuntimeException("Нет соответствующего запуска в табличке oneshot_launch_data");
            }
            LaunchStatus fromStatus = selected.getLaunchStatus();
            Set<LaunchStatus> validStatuses = VALID_TRANSITIONS.get(fromStatus);
            if (validStatuses == null) {
                throw new RuntimeException("Из этого статуса нельзя переводить");
            }
            if (!validStatuses.contains(status)) {
                throw new RuntimeException("В этот статус нельзя перевести");
            }
            validatePermissions(dslContext, selected, operator);
            selected.setLaunchStatus(status);
            oneshotLaunchDataRepository.updateStatus(dslContext, selected);
        });
    }

    private void runByLaunchId(Long launchId, LaunchStatus status, User operator) {
        StringBuilder errorMessage = new StringBuilder();
        dslContextProvider.ppcdictTransaction(c -> {
            DSLContext dslContext = c.dsl();
            var launchDataList = oneshotLaunchDataRepository.getLaunchDataByLaunchId(List.of(launchId)).get(launchId);
            if (launchDataList == null) {
                throw new RuntimeException("Нет соответствующих запусков в табличке oneshot_launch_data");
            }

            for (var launchData : launchDataList) {
                LaunchStatus fromStatus = launchData.getLaunchStatus();
                Set<LaunchStatus> validStatuses = VALID_TRANSITIONS.get(fromStatus);
                if (validStatuses == null) {
                    errorMessage.append(launchData.getId()).append(": Из этого статуса нельзя переводить; ");
                    continue;
                }
                if (!validStatuses.contains(status)) {
                    errorMessage.append(launchData.getId()).append(": В этот статус нельзя перевести; ");
                    continue;
                }
                validatePermissions(dslContext, launchData, operator);
                launchData.setLaunchStatus(status);
                oneshotLaunchDataRepository.updateStatus(dslContext, launchData);
            }
        });
        if (errorMessage.length() > 0) {
            throw new RuntimeException(errorMessage.toString());
        }

    }

    @Override
    public ValidationResult<LaunchDataInput, Defect> validate(LaunchDataInput launchDataInput) {
        ItemValidationBuilder<LaunchDataInput, Defect> vb =
                ItemValidationBuilder.of(launchDataInput, Defect.class);
        vb.check(Constraint.fromPredicate(p -> p.getLaunchId() != null || p.getLaunchDataId() != null,
                OneshotDefects.requiredAndPossiblyOnlyOneField()));
        return vb.getResult();
    }

    @Override
    protected List<LaunchDataInfo> getMassData() {
        return mapList(oneshotLaunchDataRepository.getAll(), OneshotLaunchDataTool::convert);
    }

    private static LaunchDataInfo convert(OneshotLaunchData data) {
        return new LaunchDataInfo(data.getId(),
                data.getLaunchId(),
                data.getShard(),
                data.getLaunchTime(),
                data.getLastActiveTime(),
                String.join(", ", data.getLaunchedRevisions()),
                data.getLaunchStatus(),
                data.getState(),
                data.getFinishTime(),
                getLaunchDataLogUrl(data.getLaunchTime(), data.getFinishTime(), data.getSpanId())
        );
    }

    private void validatePermissions(DSLContext dslContext, OneshotLaunchData data, User operator) {
        OneshotLaunch launch = oneshotLaunchRepository.get(dslContext, data.getLaunchId());
        if (launch == null) {
            throw new RuntimeException("Нет соответсвующего запуска в табличке oneshot_launches");
        }
        Oneshot oneshot = oneshotRepository.get(dslContext, launch.getOneshotId());
        if (oneshot == null) {
            throw new RuntimeException("Нет соответствующего ваншота в табличке oneshots");
        }

        if (oneshot.getApprovers().contains(operator.getDomainLogin())) {
            return;
        }
        if (operator.getDomainLogin().equals(launch.getLaunchCreator())) {
            return;
        }
        throw new RuntimeException("Нет прав на выполнение действий над запуском");
    }
}
