package ru.yandex.direct.internaltools.tools.oneshot.oneshot;

import java.util.Comparator;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolMassResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.tools.oneshot.ArcanumUrlUtil;
import ru.yandex.direct.internaltools.tools.oneshot.OneshotToolsUtil;
import ru.yandex.direct.internaltools.tools.oneshot.launch.OneshotLaunchesTool;
import ru.yandex.direct.internaltools.tools.oneshot.oneshot.model.OneshotInfo;
import ru.yandex.direct.internaltools.tools.oneshot.oneshot.model.OneshotInput;
import ru.yandex.direct.oneshot.core.entity.oneshot.model.ComplexOneshot;
import ru.yandex.direct.oneshot.core.entity.oneshot.service.OneshotService;
import ru.yandex.direct.oneshot.core.entity.oneshot.service.OneshotStartrekService;
import ru.yandex.direct.oneshot.core.model.Oneshot;
import ru.yandex.direct.oneshot.core.model.OneshotLaunch;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.internaltools.tools.oneshot.OneshotToolsUtil.SAFE_ONESHOT_APPROVER;
import static ru.yandex.direct.internaltools.tools.oneshot.OneshotToolsUtil.uncutOneshotClass;
import static ru.yandex.direct.oneshot.core.Constants.ONESHOT_PACKAGE;

@Tool(
        name = "Ваншоты",
        label = "oneshots",
        description = "Список ваншотов самой новой версии ваншотилки " +
                "(если одновременно работают разные версии, здесь будет список из самой новой). \n" +
                "Тильда в имени класса ваншота: \"" + ONESHOT_PACKAGE + "\"",
        consumes = OneshotInput.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.REQUEST_LAUNCH)
@Category(InternalToolCategory.ONESHOTS)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class OneshotsTool implements BaseInternalTool<OneshotInput> {

    public static final Comparator<ComplexOneshot> SORTING_COMPARATOR = Comparator.
            <ComplexOneshot>comparingInt(o -> o.getOneshot().getDeleted() ? 1 : 0)
            .thenComparingInt(o -> o.isLaunchable() ? 0 : 1)
            //.thenComparing(o -> o.getOneshot().getCreateTime(), Comparator.reverseOrder())
            .thenComparing(o -> o.getOneshot().getClassName());

    private static final int MAX_PARAM_SIZE = 65536;

    private final OneshotService oneshotService;
    private final OneshotStartrekService oneshotStartrekService;

    public OneshotsTool(OneshotService oneshotService, OneshotStartrekService oneshotStartrekService) {
        this.oneshotService = oneshotService;
        this.oneshotStartrekService = oneshotStartrekService;
    }

    @Override
    public ValidationResult<OneshotInput, Defect> validate(OneshotInput oneshotInput) {
        ComplexOneshot complexOneshot = getOneshotByClassName(oneshotInput.getClassName());
        if (complexOneshot == null) {
            throw new InternalToolValidationException("Oneshot was not found in the database: " + oneshotInput.getClassName());
        } else {
            if (complexOneshot.getOneshot().getDeleted()) {
                throw new InternalToolValidationException("This oneshot was deleted");
            }
            if (!complexOneshot.isLaunchable()) {
                throw new InternalToolValidationException("This oneshot can't be launched multiple times");
            }
        }
        String param = preprocessParam(oneshotInput.getParam());
        if (param == null) {
            return new ValidationResult<>(oneshotInput);
        }
        if (param.length() > MAX_PARAM_SIZE) {
            throw new InternalToolValidationException("JSON parameter size should be less than " + MAX_PARAM_SIZE);
        }
        try {
            JsonUtils.fromJson(param);
        } catch (IllegalArgumentException e) {
            String message = e.getMessage();

            Throwable cause = e.getCause();
            if (cause != null) {
                message = cause.getMessage();
            }
            throw new InternalToolValidationException("Invalid JSON in param: " + message);
        }
        return new ValidationResult<>(oneshotInput);
    }

    @Override
    public InternalToolMassResult<OneshotInfo> processWithoutInput() {
        var allOneshots = oneshotService.getAll().stream()
                .sorted(SORTING_COMPARATOR)
                .map(this::createOneshotInfo)
                .collect(toList());
        return new InternalToolMassResult<>(allOneshots);
    }

    @Override
    public final InternalToolMassResult<OneshotInfo> process(OneshotInput parameter) {
        ComplexOneshot complexOneshot = getOneshotByClassName(parameter.getClassName());
        String param = preprocessParam(parameter.getParam());
        String operatorLogin = parameter.getOperator().getDomainLogin();

        OneshotLaunch launch = oneshotService.launch(complexOneshot.getOneshot().getId(), operatorLogin, param);
        if (launch != null) {
            oneshotStartrekService.writeRequestLaunchComment(complexOneshot.getOneshot().getTicket(), launch.getId(),
                    operatorLogin, OneshotLaunchesTool.class.getAnnotation(Tool.class).label());
        }
        return processWithoutInput();
    }

    private ComplexOneshot getOneshotByClassName(String className) {
        return oneshotService.getByClassName(uncutOneshotClass(className));
    }

    private String preprocessParam(@Nullable String param) {
        if (param == null) {
            return null;
        }
        param = param.trim();
        if (param.length() == 0) {
            return param;
        }

        if (!param.startsWith("[") && !param.startsWith("{") && !param.startsWith("\"")) {
            return "\"" + param + "\"";
        }
        return param;
    }

    private OneshotInfo createOneshotInfo(ComplexOneshot complexOneshot) {
        Oneshot oneshot = complexOneshot.getOneshot();
        List<String> approvers = oneshot.getSafeOneshot() ? List.of(SAFE_ONESHOT_APPROVER)
                : oneshot.getApprovers().stream().sorted().collect(toList());
        return new OneshotInfo()
                .withId(oneshot.getId())
                .withClassName(OneshotToolsUtil.cutOneshotClass(oneshot.getClassName()))
                .withApprovers(approvers)
                .withTicket(oneshot.getTicket())
                .withCreateTime(oneshot.getCreateTime())
                .withMultiLaunch(oneshot.getMultiLaunch())
                .withSharded(oneshot.getSharded())
                .withSafe(oneshot.getSafeOneshot())
                .withDeleted(oneshot.getDeleted())
                .withHasLaunches(complexOneshot.hasLaunches())
                .withSourceCode(ArcanumUrlUtil.getUrl(oneshot.getClassName(), null));
    }
}
