package ru.yandex.direct.internaltools.tools.outdoor;

import java.util.Collections;
import java.util.List;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.outdoor.model.OutdoorOperator;
import ru.yandex.direct.core.entity.outdoor.repository.OutdoorOperatorRepository;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.outdoor.model.OutdoorOperatorUpdateParameter;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.validation.constraints.Constraints.validLogin;
import static ru.yandex.direct.validation.constraint.StringConstraints.matchPattern;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Tool(
        name = "Добавить/изменить оператора для наружной рекламы",
        label = "update_outdoor_operator",
        description = "Добавить/изменить оператора для наружной рекламы",
        consumes = OutdoorOperatorUpdateParameter.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.MODERATE)
@AccessGroup({InternalToolAccessRole.INTERNAL_USER})
@ParametersAreNonnullByDefault
public class OutdoorOperatorUpdateTool extends MassInternalTool<OutdoorOperatorUpdateParameter, OutdoorOperator> {
    private static final LimitOffset LIMIT_OPERATORS_PER_PAGE = LimitOffset.limited(1000);
    private static final Pattern OPERATOR_NAME_PATTERN = Pattern.compile("^[a-zA-Zа-яА-Я]+([ _][a-zA-Zа-яА-Я]+)*$");
    static final String LOGIN_FIELD = "login";
    static final String OPERATOR_NAME_FIELD = "operatorName";

    private final OutdoorOperatorRepository outdoorOperatorRepository;

    @Autowired
    public OutdoorOperatorUpdateTool(OutdoorOperatorRepository outdoorOperatorRepository) {
        this.outdoorOperatorRepository = outdoorOperatorRepository;
    }

    @Override
    public ValidationResult<OutdoorOperatorUpdateParameter, Defect> validate(OutdoorOperatorUpdateParameter params) {
        ItemValidationBuilder<OutdoorOperatorUpdateParameter, Defect> vb =
                ItemValidationBuilder.of(params, Defect.class);
        vb.item(params.getLogin(), LOGIN_FIELD)
                .check(notBlank())
                .check(validLogin(), When.isValid());

        vb.item(params.getOperatorName(), OPERATOR_NAME_FIELD)
                .check(notBlank())
                .check(matchPattern(OPERATOR_NAME_PATTERN), When.isValid());
        return vb.getResult();
    }

    @Override
    protected List<OutdoorOperator> getMassData() {
        return outdoorOperatorRepository.getAllOperators(LIMIT_OPERATORS_PER_PAGE);
    }

    @Override
    protected List<OutdoorOperator> getMassData(OutdoorOperatorUpdateParameter outdoorOperatorUpdateParameter) {
        OutdoorOperator outdoorOperator = new OutdoorOperator()
                .withLogin(outdoorOperatorUpdateParameter.getLogin())
                .withName(outdoorOperatorUpdateParameter.getOperatorName());
        outdoorOperatorRepository.addOrUpdateOperators(Collections.singletonList(outdoorOperator));
        return getMassData();
    }
}
