package ru.yandex.direct.internaltools.tools.performance;

import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterTab;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterConditionDBFormatParser;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterConditionDBFormatSerializer;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterService;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolProcessingException;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Tool(
        name = "Обновление фильтров",
        label = "update_perf_filters",
        description = "Обновление условий фильтров в смарт-группах",
        consumes = PerfFilterParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.EXPERIMENT)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.SUPERREADER, InternalToolAccessRole.MANAGER})
@ParametersAreNonnullByDefault
public class PerfFilterUpdateTool extends MassInternalTool<PerfFilterParameters, IntToolPerfFilter> {

    private final ShardHelper shardHelper;
    private final PerformanceFilterService performanceFilterService;
    private final PerformanceFilterValidationService performanceFilterValidationService;
    private final FeatureService featureService;

    private Long perfFilterId;
    private Long adGroupId;
    private ClientId clientId;
    private String oldConditionsJson;
    private String newConditionsJson;
    private Long operatorUid;

    public PerfFilterUpdateTool(ShardHelper shardHelper,
                                PerformanceFilterService performanceFilterService,
                                PerformanceFilterValidationService performanceFilterValidationService,
                                FeatureService featureService) {
        this.shardHelper = shardHelper;
        this.performanceFilterService = performanceFilterService;
        this.performanceFilterValidationService = performanceFilterValidationService;
        this.featureService = featureService;
    }

    @Override
    public ValidationResult<PerfFilterParameters, Defect> validate(PerfFilterParameters params) {
        ItemValidationBuilder<PerfFilterParameters, Defect> vb = ItemValidationBuilder.of(params, Defect.class);
        vb.item(params.getAdGroupId(), "adGroupId")
                .check(notNull());
        vb.item(params.getPerfFilterId(), "perfFilterId")
                .check(notNull());
        vb.item(params.getConditionJson(), "conditionJson")
                .check(notNull())
                .check(notBlank());
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }
        try {
            JsonUtils.fromJson(params.getConditionJson());
        } catch (IllegalArgumentException e) {
            String message = e.getMessage();

            Throwable cause = e.getCause();
            if (cause != null) {
                message = cause.getMessage();
            }
            throw new InternalToolValidationException("Invalid JSON in param: " + message);
        }
        return vb.getResult();
    }

    @Override
    protected List<IntToolPerfFilter> getMassData(PerfFilterParameters params) {
        perfFilterId = params.getPerfFilterId();
        adGroupId = params.getAdGroupId();
        clientId = getClientIdFromLogin(params);
        if (!featureService
                .isEnabledForClientId(clientId, FeatureName.CHANGE_FILTER_CONDITIONS_ALLOWED)) {
            throw new InternalToolProcessingException("Client " + clientId.asLong()
                    + " must have feature CHANGE_FILTER_CONDITIONS_ALLOWED");
        }

        String conditionJson = params.getConditionJson();
        operatorUid = getUidFromLogin(params);
        PerformanceFilter filter = getFilterIfExisted();
        oldConditionsJson = getConditionsInString(filter);

        List<PerformanceFilterCondition> updatedPerformanceFilterConditions =
                performanceFilterService.parseConditionJson(filter, conditionJson);
        PerformanceFilterTab updatedTab = getUpdatedTab(updatedPerformanceFilterConditions);
        filter.setConditions(updatedPerformanceFilterConditions);
        if (!updatedTab.equals(filter.getTab())) {
            filter.setTab(updatedTab);
        }

        ValidationResult<List<PerformanceFilter>, Defect> validationResult =
                performanceFilterValidationService.validate(clientId, operatorUid, singletonList(filter));
        if (validationResult.hasAnyErrors()) {
            throw new InternalToolProcessingException("Ошибка при валидации фильтра: "
                    + validationResult.flattenErrors());
        }
        performanceFilterService.updateConditionJsonField(clientId, adGroupId, perfFilterId, conditionJson);
        performanceFilterService.updatePerformanceFilterTab(clientId, perfFilterId, updatedTab);
        newConditionsJson = conditionJson;
        return getMassData();
    }

    private PerformanceFilterTab getUpdatedTab(List<PerformanceFilterCondition> conditions) {
        if (conditions.isEmpty()) {
            return PerformanceFilterTab.ALL_PRODUCTS;
        }
        if (PerformanceFilterConditionDBFormatParser.isSupportedByTreeTab(conditions)) {
            return PerformanceFilterTab.TREE;
        }
        return PerformanceFilterTab.CONDITION;
    }

    /**
     * Для валидации фильтров необходим operatorUid с доступом к изменению кампаний.
     * Используем для этого uid оператора внутреннего отчета или введенный логин.
     */
    private Long getUidFromLogin(PerfFilterParameters params) {
        if (params.getLogin() == null) {
            return params.getOperator().getUid();
        }
        return shardHelper.getUidByLogin(params.getLogin());
    }

    private ClientId getClientIdFromLogin(PerfFilterParameters params) {
        if (params.getLogin() == null) {
            return params.getOperator().getClientId();
        }
        return ClientId.fromLong(shardHelper.getClientIdByLogin(params.getLogin()));
    }

    private PerformanceFilter getFilterIfExisted() {
        Map<Long, List<PerformanceFilter>> performanceFilters = performanceFilterService
                .getPerformanceFilters(clientId, singletonList(adGroupId));
        checkState(performanceFilters.containsKey(adGroupId), "Группа у клиента "
                + shardHelper.getLoginByUid(operatorUid) + " не найдена или не содержит фильтров");
        PerformanceFilter existedFilter = StreamEx.of(performanceFilters.get(adGroupId))
                .findFirst(f -> f.getId().equals(perfFilterId))
                .orElse(null);
        checkState(existedFilter != null, "Фильтр с указанным id не найден");
        return existedFilter;
    }

    @Nullable
    @Override
    protected List<IntToolPerfFilter> getMassData() {
        if (perfFilterId == null) {
            return null;
        }
        IntToolPerfFilter intToolPerfFilter = new IntToolPerfFilter()
                .withAdGroupId(adGroupId)
                .withPerfFilterId(perfFilterId)
                .withNewConditionsJson(newConditionsJson)
                .withOldConditionsJson(oldConditionsJson);
        return singletonList(intToolPerfFilter);
    }

    private static String getConditionsInString(PerformanceFilter filter) {
        return PerformanceFilterConditionDBFormatSerializer.INSTANCE.serialize(filter.getConditions());
    }

}
