package ru.yandex.direct.internaltools.tools.pitags;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupBsTagsService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.utils.StringUtils;

import static ru.yandex.direct.internaltools.tools.pitags.PiTagsParameter.Mode.ADD;

@Tool(
        name = "ПИ тэги",
        label = "adgroup_pi_tags",
        description = "Добавление, обновление и удаление ПИ-тэгов на группе",
        consumes = PiTagsParameter.class,
        type = InternalToolType.REPORT
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.OTHER)
public class AdGroupPiTagsTool extends MassInternalTool<PiTagsParameter, InfoTags> {

    private final AdGroupBsTagsService adGroupBsTagsService;
    private final AdGroupRepository adGroupRepository;
    private final ShardHelper shardHelper;
    private final UserService userService;

    @Autowired
    public AdGroupPiTagsTool(AdGroupBsTagsService adGroupBsTagsService, AdGroupRepository adGroupRepository,
                             ShardHelper shardHelper, UserService userService) {
        this.adGroupBsTagsService = adGroupBsTagsService;
        this.adGroupRepository = adGroupRepository;
        this.shardHelper = shardHelper;
        this.userService = userService;
    }

    @Override
    protected List<InfoTags> getMassData(PiTagsParameter parameter) {
        List<Long> adgroupIds = new ArrayList<>();

        if (!parameter.getCids().isBlank()) {
            List<Long> cids = StringUtils.parseLongsFromString(parameter.getCids());

            shardHelper.forEachShard(s -> {
                for (var pids : adGroupRepository.getAdGroupIdsByCampaignIds(s, cids).values()) {
                    adgroupIds.addAll(pids);
                }
            });
        }

        if (!parameter.getClientIds().isBlank()) {
            List<Long> clientIds = StringUtils.parseLongsFromString(parameter.getClientIds());
            adgroupIds.addAll(adgroupIdsByClientId(clientIds));
        }

        if (!parameter.getLogins().isBlank()) {
            List<String> logins = splitString(parameter.getLogins());
            Collection<Long> clientIds = userService.massGetClientIdsByLogin(logins).values();
            adgroupIds.addAll(adgroupIdsByClientId(clientIds));
        }

        if (!parameter.getPids().isBlank()) {
            adgroupIds.addAll(StringUtils.parseLongsFromString(parameter.getPids()));
        }

        String pageTag = "OrderTag".equals(parameter.getTagType()) ? parameter.getTag() : null;
        String targetTag = "TargetTag".equals(parameter.getTagType()) ? parameter.getTag() : null;

        switch (parameter.getMode()) {
            case ADD:
            case REPLACE:
                shardHelper.groupByShard(adgroupIds, ShardKey.PID, Function.identity())
                        .chunkedBy(1000)
                        .forEach((shard, pids) -> adGroupBsTagsService.setRawTagsByAdgroupIds(shard, pageTag,
                                targetTag, pids,
                                parameter.getMode() == ADD));
                break;
            case REMOVE:
                shardHelper.groupByShard(adgroupIds, ShardKey.PID, Function.identity())
                        .chunkedBy(1000)
                        .forEach((shard, pids) -> adGroupBsTagsService.removeRawTagsByAdgroupIds(shard, pageTag,
                                targetTag, pids));
                break;
        }

        return List.of(new InfoTags((long) adgroupIds.size()));
    }

    private List<Long> adgroupIdsByClientId(Collection<Long> clientIds) {
        List<Long> result = new ArrayList<>();

        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID, Function.identity())
                .chunkedBy(10)
                .forEach((shard, cls) -> {
                    var pids = adGroupRepository.getAdGroupIdsByClientIds(shard, cls);
                    result.addAll(pids);
                });

        return result;
    }

    private List<String> splitString(String s) {
        return Stream.of(s.split(","))
                .map(String::trim)
                .collect(Collectors.toList());
    }
}
