package ru.yandex.direct.internaltools.tools.ppcproperties;

import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.ppcproperties.container.DeprecatedReportToToolLink;
import ru.yandex.direct.internaltools.tools.ppcproperties.container.PropertyInfo;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notEmpty;

@Tool(
        name = "Пометка внутреннего отчета как устаревшего",
        label = "set_report_deprecated",
        description = "Установка ppc_property, который сообщит в перловый Директ о том, что появилась новая версия "
                + "инструмента. Перловый Директ сообщит об этом пользователям в интерфейсе",
        consumes = DeprecatedReportToToolLink.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.SET)
@Category(InternalToolCategory.PROPERTIES)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class SetOldReportDeprecatedTool extends MassInternalTool<DeprecatedReportToToolLink, PropertyInfo> {
    /**
     * Константа продублирована в перле в InternalReports::REPORT_NEW_VERSION_PROPERTY_PREFIX.
     * При редактирвоании здесь, нужно поправить и там
     */
    private static final String PROPERTY_PREFIX = "INTERNAL_REPORT_NEW_VERSION_";

    private final PpcPropertiesSupport propertiesSupport;

    @Autowired
    public SetOldReportDeprecatedTool(PpcPropertiesSupport propertiesSupport) {
        this.propertiesSupport = propertiesSupport;
    }

    @Override
    public ValidationResult<DeprecatedReportToToolLink, Defect> validate(DeprecatedReportToToolLink param) {
        ItemValidationBuilder<DeprecatedReportToToolLink, Defect> validationBuilder =
                ItemValidationBuilder.of(param);

        validationBuilder
                .item(param.getOldReport(), "oldReport")
                .check(notNull())
                .check(notEmpty())
                .check(maxStringLength(PpcPropertiesSupport.NAME_MAX_LENGTH - PROPERTY_PREFIX.length()));

        return validationBuilder.getResult();
    }

    @Override
    protected List<PropertyInfo> getMassData() {
        return propertiesSupport.getByPrefix(PROPERTY_PREFIX)
                .entrySet().stream()
                .map(e -> new PropertyInfo(e.getKey().substring(PROPERTY_PREFIX.length()), e.getValue()))
                .sorted(Comparator.comparing(PropertyInfo::getName))
                .collect(Collectors.toList());
    }

    @Override
    protected List<PropertyInfo> getMassData(DeprecatedReportToToolLink param) {
        propertiesSupport.set(PROPERTY_PREFIX + param.getOldReport(), param.getNewTool());
        return getMassData();
    }
}
