package ru.yandex.direct.internaltools.tools.ppcproperties;

import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum;
import ru.yandex.direct.core.entity.ppcproperty.model.WebEditablePpcProperty;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.ppcproperties.container.NewPropertyValue;
import ru.yandex.direct.internaltools.tools.ppcproperties.container.PropertyInfo;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.internaltools.tools.ppcproperties.SetTypedPropertyTool.DEFINED_AND_NOT_BLANK;
import static ru.yandex.direct.internaltools.tools.ppcproperties.SetTypedPropertyTool.NOT_DEFINED_OR_BLANK;
import static ru.yandex.direct.internaltools.tools.ppcproperties.container.NewTypedPropertyValue.VALUE_FIELD_NAME;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.When.isFalse;
import static ru.yandex.direct.validation.builder.When.isTrue;

@Tool(
        name = "Ограниченное редактирование ppc_property",
        label = "set_ppc_property",
        description = "Интерфейс для редактирования тех ppc_property, для которых есть разрешение.\n"
                + "Также позволяет удалить свойство из базы.",
        consumes = NewPropertyValue.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.SET)
@Category(InternalToolCategory.PROPERTIES)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class SetPropertyTool extends MassInternalTool<NewPropertyValue, PropertyInfo> {
    private static final List<String> ALLOWED_PROPS =
            mapList(Arrays.asList(WebEditablePpcProperty.ALLOWED_TO_EDIT.toArray(new PpcPropertyEnum[]{})),
                    PpcPropertyEnum::getName);

    private final PpcPropertiesSupport propertiesSupport;

    @Autowired
    public SetPropertyTool(PpcPropertiesSupport propertiesSupport) {
        this.propertiesSupport = propertiesSupport;
    }

    @Override
    protected List<PropertyInfo> getMassData() {
        return propertiesSupport.getByNames(ALLOWED_PROPS)
                .entrySet().stream()
                .map(e -> new PropertyInfo(e.getKey(), e.getValue()))
                .sorted(Comparator.comparing(PropertyInfo::getName))
                .collect(Collectors.toList());
    }

    @Override
    public ValidationResult<NewPropertyValue, Defect> validate(NewPropertyValue newPropertyValue) {
        Boolean toRemove = newPropertyValue.getToRemove();

        ItemValidationBuilder<NewPropertyValue, Defect> vb = ItemValidationBuilder.of(newPropertyValue);
        vb.item(newPropertyValue.getValue(), VALUE_FIELD_NAME)
                .check(DEFINED_AND_NOT_BLANK, isFalse(toRemove))
                .check(NOT_DEFINED_OR_BLANK, isTrue(toRemove));
        return vb.getResult();
    }

    @Override
    protected List<PropertyInfo> getMassData(NewPropertyValue param) {
        if (param.getToRemove()) {
            propertiesSupport.remove(param.getName().getName());
        } else {
            propertiesSupport.set(param.getName().getName(), param.getValue());
        }
        return getMassData();
    }
}
