package ru.yandex.direct.internaltools.tools.ppcproperties;

import java.util.Comparator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyData;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.common.db.PpcPropertyParseException;
import ru.yandex.direct.common.db.PpcPropertyType;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.ppcproperties.container.NewTypedPropertyValue;
import ru.yandex.direct.internaltools.tools.ppcproperties.container.TypedPropertyInfo;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.common.db.PpcPropertyType.UNSUPPORTED_TYPE;
import static ru.yandex.direct.core.entity.ppcproperty.model.WebEditablePpcProperty.TYPED_PROPS_ALLOWED_TO_EDIT;
import static ru.yandex.direct.internaltools.tools.ppcproperties.container.NewTypedPropertyValue.VALUE_FIELD_NAME;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.builder.When.isFalse;
import static ru.yandex.direct.validation.builder.When.isTrue;
import static ru.yandex.direct.validation.builder.When.isValid;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidFormat;
import static ru.yandex.direct.validation.defect.CommonDefects.mustBeEmpty;
import static ru.yandex.direct.validation.defect.StringDefects.notEmptyString;

@Tool(
        name = "Ограниченное редактирование типизированных свойств",
        label = "set_typed_ppc_property",
        description = "Интерфейс для редактирования ограниченного списка типизированных свойств (PpcPropertyName).\n"
                + "Отображает десериализованные значения или \"" + UNSUPPORTED_TYPE + "\" при ошибке парсинга.\n"
                + "При сохранении значения проверяется корректность формата (для некоторых типов: чисел и их коллекций,"
                + "даты/времени), в базу записывается сериализованное значение.\n"
                + "Для типа ShardSet можно указать 'all' (без кавычек) — сохранится список из всех шардов.\n\n"
                + "Также позволяет удалить свойство из базы.",
        consumes = NewTypedPropertyValue.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.SET)
@Category(InternalToolCategory.PROPERTIES)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class SetTypedPropertyTool extends MassInternalTool<NewTypedPropertyValue, TypedPropertyInfo> {

    private static final Map<String, PpcPropertyName> PROPERTIES = StreamEx.of(TYPED_PROPS_ALLOWED_TO_EDIT)
            .mapToEntry(PpcPropertyName::getName, Function.identity())
            .toImmutableMap();

    static final Constraint<String, Defect> DEFINED_AND_NOT_BLANK =
            Constraint.fromPredicateOfNullable(StringUtils::isNotBlank, notEmptyString());
    static final Constraint<String, Defect> NOT_DEFINED_OR_BLANK =
            Constraint.fromPredicateOfNullable(StringUtils::isAnyBlank, mustBeEmpty());


    private final PpcPropertiesSupport propertiesSupport;
    private final ShardHelper shardHelper;

    @Autowired
    public SetTypedPropertyTool(PpcPropertiesSupport propertiesSupport, ShardHelper shardHelper) {
        this.propertiesSupport = propertiesSupport;
        this.shardHelper = shardHelper;
    }

    @Override
    protected List<TypedPropertyInfo> getMassData() {
        return EntryStream.of(propertiesSupport.getFullByNames(PROPERTIES.keySet()))
                .mapKeys(PROPERTIES::get)
                .mapToValue(SetTypedPropertyTool::parseTypedValue)
                .mapKeyValue(TypedPropertyInfo::new)
                .sorted(Comparator.comparing(TypedPropertyInfo::getName))
                .collect(Collectors.toList());
    }

    @Override
    public ValidationResult<NewTypedPropertyValue, Defect> validate(NewTypedPropertyValue newTypedPropertyValue) {
        PpcPropertyType<?> type = PROPERTIES.get(newTypedPropertyValue.getName()).getType();
        Predicate<String> canBeSerialized = canBeSerialized(type);

        Boolean toRemove = newTypedPropertyValue.getToRemove();

        ItemValidationBuilder<NewTypedPropertyValue, Defect> vb = ItemValidationBuilder.of(newTypedPropertyValue);
        vb.item(newTypedPropertyValue.getValue(), VALUE_FIELD_NAME)
                .check(DEFINED_AND_NOT_BLANK, isFalse(toRemove))
                .check(NOT_DEFINED_OR_BLANK, isTrue(toRemove))
                .check(fromPredicate(canBeSerialized, invalidFormat()), isValid());
        return vb.getResult();
    }

    @Override
    protected List<TypedPropertyInfo> getMassData(NewTypedPropertyValue param) {
        if (param.getToRemove()) {
            propertiesSupport.remove(param.getName());
        } else {
            propertiesSupport.set(param.getName(), serialize(param));
        }
        return getMassData();
    }

    private String serialize(NewTypedPropertyValue param) {
        PpcPropertyType<?> type = PROPERTIES.get(param.getName()).getType();
        return serialize(type, param.getValue());
    }

    private String serialize(PpcPropertyType<?> type, String value) {
        if (type instanceof PpcPropertyType.ShardSet
                && "all".equals(value)) {
            LinkedHashSet<Integer> shards = StreamEx.of(shardHelper.dbShards()).toCollection(LinkedHashSet::new);
            return ((PpcPropertyType.IntegerSet) type).serialize(shards);
        }
        return type.reSerialize(value);
    }

    private Predicate<String> canBeSerialized(PpcPropertyType<?> type) {
        return value -> {
            try {
                return serialize(type, value) != null;
            } catch (PpcPropertyParseException e) {
                return false;
            }
        };
    }

    private static PpcPropertyData<?> parseTypedValue(PpcPropertyName<?> ppcPropertyName,
                                                   @Nullable PpcPropertyData<String> ppcPropertyData) {
        String value = ppcPropertyData != null ? ppcPropertyData.getValue() : null;
        if (value == null) {
            return ppcPropertyData;
        }
        Object newValue;
        try {
            newValue = ppcPropertyName.getType().deserialize(value);
        } catch (PpcPropertyParseException e) {
            newValue = e.getMessage();
        }
        return new PpcPropertyData<>(newValue, ppcPropertyData.getLastChange());
    }
}
