package ru.yandex.direct.internaltools.tools.pricesales;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.DecimalFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.tools.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CpmPriceCampaign;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusApprove;
import ru.yandex.direct.core.entity.campaign.model.WalletRestMoney;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignOperationService;
import ru.yandex.direct.core.entity.campaign.service.CampaignOptions;
import ru.yandex.direct.core.entity.campaign.service.CpmPriceCampaignService;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageBase;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageService;
import ru.yandex.direct.core.entity.product.model.Product;
import ru.yandex.direct.core.entity.product.repository.ProductRepository;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.model.RuleType;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsCpmPriceApprover;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardSupport;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.RequiredFeature;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.pricesales.container.PriceSalesCampaignInfo;
import ru.yandex.direct.internaltools.tools.pricesales.model.PriceSalesCampaignsListAndApproveParameter;
import ru.yandex.direct.inventori.InventoriCampaignsPredictionClient;
import ru.yandex.direct.inventori.InventoriGeneralCampaignsPredictionResponse;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.Region;
import ru.yandex.direct.result.Result;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.PricePackageValidator.REGION_TYPE_REGION;
import static ru.yandex.direct.dbschema.ppc.tables.CampaignsCpmPrice.CAMPAIGNS_CPM_PRICE;
import static ru.yandex.direct.feature.FeatureName.IS_ABLE_TO_APPROVE_CAMPAIGNS;
import static ru.yandex.direct.inventori.InventoriGeneralCampaignsPredictionResponse.TRAFFIC_LIGHT_GREEN;
import static ru.yandex.direct.utils.CollectionUtils.flatToList;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Tool(
        name = "Подтверждение бронирования для Главной страницы",
        label = "price_sales_campaigns_list",
        description = "Редактирование статуса апрува в кампаниях прайсовых продаж и список кампаний ожидающих апрува.",
        consumes = PriceSalesCampaignsListAndApproveParameter.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.PRICE_SALES)
@RequiredFeature(IS_ABLE_TO_APPROVE_CAMPAIGNS)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.MANAGER,
        InternalToolAccessRole.SUPERREADER, InternalToolAccessRole.SUPPORT})
@ParametersAreNonnullByDefault
public class PriceSalesCampaignsListAndApproveTool extends MassInternalTool<PriceSalesCampaignsListAndApproveParameter,
        PriceSalesCampaignInfo> {

    private static final Logger logger = LoggerFactory.getLogger(PriceSalesCampaignsListAndApproveTool.class);
    private static final DecimalFormat MONEY_FORMATTER = createMoneyFormatter();

    private final ShardSupport shardSupport;
    private final CpmPriceCampaignService cpmPriceCampaignService;
    private final PricePackageRepository pricePackageRepository;
    private final UserRepository userRepository;
    private final CampaignRepository campaignRepository;
    private final CampaignOperationService campaignOperationService;
    private final PricePackageService pricePackageService;
    private final AdGroupRepository adGroupRepository;
    private final RetargetingConditionRepository retConditionRepository;
    private final CryptaSegmentRepository cryptaSegmentRepository;
    private final ClientGeoService clientGeoService;
    private final ProductRepository productRepository;
    private final DslContextProvider dslContextProvider;
    private final InventoriCampaignsPredictionClient inventoryClient;

    @Autowired
    public PriceSalesCampaignsListAndApproveTool(
            ShardSupport shardSupport,
            CpmPriceCampaignService cpmPriceCampaignService,
            PricePackageRepository pricePackageRepository,
            UserRepository userRepository,
            CampaignRepository campaignRepository,
            CampaignOperationService campaignOperationService,
            PricePackageService pricePackageService,
            AdGroupRepository adGroupRepository,
            RetargetingConditionRepository retConditionRepository,
            CryptaSegmentRepository cryptaSegmentRepository,
            ClientGeoService clientGeoService,
            ProductRepository productRepository,
            DslContextProvider dslContextProvider,
            InventoriCampaignsPredictionClient inventoryClient) {
        this.shardSupport = shardSupport;
        this.cpmPriceCampaignService = cpmPriceCampaignService;
        this.pricePackageRepository = pricePackageRepository;
        this.userRepository = userRepository;
        this.campaignRepository = campaignRepository;
        this.campaignOperationService = campaignOperationService;
        this.pricePackageService = pricePackageService;
        this.adGroupRepository = adGroupRepository;
        this.retConditionRepository = retConditionRepository;
        this.cryptaSegmentRepository = cryptaSegmentRepository;
        this.clientGeoService = clientGeoService;
        this.productRepository = productRepository;
        this.dslContextProvider = dslContextProvider;
        this.inventoryClient = inventoryClient;
    }

    @Override
    protected List<PriceSalesCampaignInfo> getMassData(PriceSalesCampaignsListAndApproveParameter parameter) {
        Long campaignId = parameter.getCampaignId();
        Long operatorUid = parameter.getOperator().getUid();
        Integer shard = shardSupport.getValue(ShardKey.CID, campaignId, ShardKey.SHARD, Integer.class);
        if (shard == null) {
            throw new InternalToolValidationException(String.format("Campaign %d is not found", campaignId));
        }

        UidAndClientId uidAndClientId = getCampaignUidAndClientId(shard, campaignId);
        ModelChanges<CpmPriceCampaign> statusApproveChange = ModelChanges.build(campaignId, CpmPriceCampaign.class,
                CpmPriceCampaign.FLIGHT_STATUS_APPROVE, parameter.getStatusApprove());

        var options = new CampaignOptions();
        Result<Long> result = campaignOperationService.createRestrictedCampaignUpdateOperation(
                        singletonList(statusApproveChange), operatorUid, uidAndClientId, options)
                .apply()
                .get(0);
        if (!result.isSuccessful()) {
            throw new InternalToolValidationException(
                    String.format("Failed to update campaign %d status approve. Validation error.", campaignId))
                    .withValidationResult(result.getValidationResult());
        }
        updateCampaignsApproverData(shard, parameter.getCampaignId(), parameter.getStatusApprove());
        return getMassData();
    }

    private UidAndClientId getCampaignUidAndClientId(int shard, Long campaignId) {
        Campaign campaign = campaignRepository.getCampaigns(shard, singleton(campaignId)).get(0);
        if (campaign.getType() != CampaignType.CPM_PRICE) {
            throw new InternalToolValidationException(String.format("Campaign %d is not cpm price", campaignId));
        }

        return UidAndClientId.of(campaign.getUserId(), ClientId.fromLong(campaign.getClientId()));
    }

    private void updateCampaignsApproverData(int shard, Long campaignId, PriceFlightStatusApprove statusApprove) {
        if (campaignId == null) {
            return;
        }

        int rows = dslContextProvider.ppc(shard)
                .update(CAMPAIGNS_CPM_PRICE)
                // даже если менеджер ставит статус аппрува New, то все равно будет полезно записывать, чтобы понимать,
                // что это не снуля проставленный статус
                .set(CAMPAIGNS_CPM_PRICE.APPROVER, CampaignsCpmPriceApprover.manager)
                .set(CAMPAIGNS_CPM_PRICE.APPROVE_DATE, LocalDateTime.now())
                .where(CAMPAIGNS_CPM_PRICE.CID.eq(campaignId))
                .execute();

        if (rows > 0) {
            logger.info(String.format("CampaignsCpmPrice: " +
                            "{\"campaignIds\": [%s], \"statusApprove\": \"%s\", \"approver\": \"%s\"}",
                    campaignId, statusApprove.name(), CampaignsCpmPriceApprover.manager.getLiteral()));
        } else {
            throw new InternalToolValidationException(
                    String.format("Failed to update campaign %d approver data.", campaignId));
        }
    }

    @Override
    protected List<PriceSalesCampaignInfo> getMassData() {
        Map<Integer, List<CpmPriceCampaign>> campaignsByShard =
                cpmPriceCampaignService.getCpmPriceCampaignsWaitingApproveByShard();
        excludeOddCampaigns(campaignsByShard);
        List<CpmPriceCampaign> campaigns = flatToList(campaignsByShard.values());
        if (campaigns.isEmpty()) {
            return emptyList();
        }

        Set<Long> pricePackageIds = listToSet(campaigns, CpmPriceCampaign::getPricePackageId);
        Map<Long, PricePackage> pricePackages = pricePackageRepository.getPricePackages(pricePackageIds);

        Map<Long, User> users = getClientAndAgencyUsers(campaignsByShard);

        Map<Long, WalletRestMoney> walletsRestMoney = cpmPriceCampaignService.getWalletRestMoneyMap(campaignsByShard);
        Map<Long, String> targetingDefaultAdGroup = getTargetingDefaultAdGroup(campaignsByShard);

        var geoString = geoString(pricePackages, campaignsByShard);
        var productIds =
                pricePackages.values().stream().map(PricePackageBase::getProductId).distinct().collect(Collectors.toList());

        var products = productRepository.getProductsById(productIds).stream()
                .collect(Collectors.toMap(Product::getId, e -> e));


        return campaigns.stream()
                .map(campaign -> {
                    Long campaignId = campaign.getId();
                    Long pricePackageId = campaign.getPricePackageId();
                    PricePackage pricePackage = pricePackages.get(pricePackageId);
                    checkNotNull(pricePackage, "pricePackage id %s doesn't exist but used in " +
                            "campaign with id %s", pricePackageId, campaignId);
                    User clientUser = users.get(campaign.getUid());
                    User agencyUser = ifNotNull(campaign.getAgencyUid(), users::get);
                    WalletRestMoney walletRestMoney = walletsRestMoney.get(campaign.getWalletId());

                    String businessUnitName = ifNotNull(products.get(pricePackage.getProductId()),
                            Product::getBusinessUnitName);

                    var frequencyLimit = campaign.getImpressionRateCount();
                    var frequencyLimitInterval = nvl(campaign.getImpressionRateIntervalDays(), 0);

                    if (frequencyLimit == null && pricePackage.getCampaignOptions() != null && pricePackage.getCampaignOptions().getShowsFrequencyLimit() != null) {
                        frequencyLimit = pricePackage.getCampaignOptions().getShowsFrequencyLimit().getFrequencyLimit();

                        if (pricePackage.getCampaignOptions().getShowsFrequencyLimit().getFrequencyLimitIsForCampaignTime()) {
                            frequencyLimitInterval = 0;
                        } else {
                            frequencyLimitInterval =
                                    pricePackage.getCampaignOptions().getShowsFrequencyLimit().getFrequencyLimitDays();
                        }
                    }
                    InventoriGeneralCampaignsPredictionResponse campaignPrediction;

                    InventoriColor inventoriColor;
                    if (campaign.getEndDate() == null || campaign.getEndDate().isBefore(LocalDate.now())) {
                        inventoriColor = InventoriColor.RED;
                    } else {
                        try {
                            campaignPrediction = inventoryClient.generalCampaignsPrediction(campaignId);
                            if (campaignPrediction.getTrafficLightColour().equals(TRAFFIC_LIGHT_GREEN)) {
                                inventoriColor = InventoriColor.GREEN;
                            } else {
                                inventoriColor = InventoriColor.RED;
                            }
                        } catch (Exception e) {
                            inventoriColor = InventoriColor.ERROR;
                            logger.error("Inventory request failed for cid: {}, cause: {}", campaignId, e.getCause());
                        }
                    }
                    return new PriceSalesCampaignInfo()
                            .withCampaignId(campaignId)
                            .withStartDate(campaign.getStartDate())
                            .withEndDate(campaign.getEndDate())
                            .withOrderVolume(campaign.getFlightOrderVolume())
                            .withPricePackageTitle(pricePackage.getTitle() + " (" + pricePackageId + ")")
                            .withClientId(campaign.getClientId())
                            .withClientLogin(clientUser.getLogin())
                            .withAgencyClientId(campaign.getAgencyId())
                            .withAgencyLogin(ifNotNull(agencyUser, User::getLogin))
                            .withWalletMoney(ifNotNull(walletRestMoney, this::extractMoneyAsText))
                            .withGeo(geoString.get(campaignId))
                            .withBusinessUnitName(businessUnitName)
                            .withFrequencyLimit(frequencyLimit)
                            .withImpressionRateIntervalDays(frequencyLimitInterval)
                            .withMinVolume(pricePackage.getOrderVolumeMin())
                            .withTargeting(targetingDefaultAdGroup.getOrDefault(campaignId,
                                    "Нет дефолтной группы"))
                            .withBrandlift(fieldBrandlift(campaign))
                            .withAvgDayBudget(fieldAvgDayBudget(campaign, pricePackage))
                            .withInventoriColor(inventoriColor.getText());
                })
                .collect(Collectors.toList());
    }

    /**
     * Возвращает "Регионы показов" для отображения в таблице
     */
    private Map<Long, String> geoString(Map<Long, PricePackage> pricePackages,
                                        Map<Integer, List<CpmPriceCampaign>> campaignsByShard) {
        return EntryStream.of(campaignsByShard)
                .mapKeyValue((shard, campaignsInShard) -> geoString(shard, campaignsInShard, pricePackages))
                .flatMapToEntry(identity())
                .toMap();
    }

    private Map<Long, Long> getCampaignIdToPid(int shard, List<Long> campaignIds) {
        //Регионы показов читаем с дефолтной группы
        var defAdGroupIdByCampaignId = adGroupRepository.getDefaultPriceSalesAdGroupIdByCampaignId(shard, campaignIds);

        //если нет дефольной, то берём все
        var campaignIdToAdGroupIds = adGroupRepository.getAdGroupIdsByCampaignIds(shard, campaignIds);

        return StreamEx.of(campaignIds)
                .filter(campaignIdToAdGroupIds::containsKey)
                .toMap(identity(),
                        cid -> {
                            if (defAdGroupIdByCampaignId.containsKey(cid)) {
                                return defAdGroupIdByCampaignId.get(cid);
                            }
                            if (campaignIdToAdGroupIds.containsKey(cid) && !campaignIdToAdGroupIds.get(cid).isEmpty()) {
                                return campaignIdToAdGroupIds.get(cid).get(0);
                            }
                            return null;
                        });
    }

    private Map<Long, String> geoString(int shard, List<CpmPriceCampaign> campaignsInShard,
                                        Map<Long, PricePackage> pricePackages) {
        var campaignIds = StreamEx.of(campaignsInShard)
                .map(CpmPriceCampaign::getId)
                .toList();

        Map<Long, Long> campaignIdToPid = getCampaignIdToPid(shard, campaignIds);

        //и вытягиваем сами данные
        var adGroups = StreamEx.of(adGroupRepository.getAdGroups(shard,
                StreamEx.of(campaignIdToPid.values()).filter(it -> it != null).toList()))
                .groupingBy(AdGroup::getCampaignId);

        return StreamEx.of(campaignsInShard)
                .toMap(CpmPriceCampaign::getId,
                        campaign -> {
                            var cid = campaign.getId();
                            var pricePackage = pricePackages.get(campaign.getPricePackageId());
                            var adGroup = adGroups.get(cid);
                            if (adGroup == null || adGroup.isEmpty()) {
                                return geoString(nvl(pricePackage.getTargetingsCustom().getGeo(),
                                        pricePackage.getTargetingsFixed().getGeo()),
                                        geoType(pricePackage), campaign);
                            }
                            return geoString(adGroup.get(0).getGeo(), geoType(pricePackage), campaign);
                        });
    }

    private Integer geoType(PricePackage pricePackage) {
        return nvl(pricePackage.getTargetingsCustom().getGeoType(), pricePackage.getTargetingsFixed().getGeoType());
    }

    private String geoString(List<Long> geo, Integer geoType, CpmPriceCampaign campaign) {
        int maxSize = 10;
        var geoExpanded = geoType.equals(REGION_TYPE_REGION) ?
                geo : campaign.getFlightTargetingsSnapshot().getGeoExpanded();
        geoExpanded = clientGeoService.convertForWeb(geoExpanded, getPriceGeoTree());
        var geoString = geoExpanded.stream()
                .map(getPriceGeoTree()::getRegion)
                .filter(Objects::nonNull)
                .map(Region::getNameRu)
                .limit(maxSize)
                .collect(Collectors.joining(", "));
        if (geoExpanded.size() > maxSize) {
            geoString += " и ещё " + (geoExpanded.size() - maxSize);
        }
        return geoString;
    }

    /**
     * Выбрасываем кампании, которые находятся не на шарде пользователя, к которому они привязаны. Такое может случиться
     * на devtest после телепортации, мы хотим чтобы tool продолжал работать на такой неправильной базе.
     */
    private void excludeOddCampaigns(Map<Integer, List<CpmPriceCampaign>> campaignsByShard) {
        List<Long> uids = campaignsByShard.values().stream()
                .flatMap(List::stream)
                .map(CpmPriceCampaign::getUid)
                .distinct()
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        List<Integer> uidShards = shardSupport.getShards(ShardKey.UID, uids);

        Map<Long, Integer> shardByUids = EntryStream.zip(uids, uidShards).toMap();

        campaignsByShard.forEach((campaignShard, campaigns) ->
                campaigns.removeIf(c -> {
                    Integer shard = shardByUids.get(c.getUid());
                    return !campaignShard.equals(shard);
                }));
    }

    private Map<Long, User> getClientAndAgencyUsers(Map<Integer, List<CpmPriceCampaign>> campaignsByShard) {
        List<Long> agencyUids = campaignsByShard.values().stream()
                .flatMap(List::stream)
                .map(CpmPriceCampaign::getAgencyUid)
                .distinct()
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        List<Integer> agencyShards = shardSupport.getShards(ShardKey.UID, agencyUids);

        EntryStream<Integer, Long> campaignUidsStream = EntryStream.of(campaignsByShard)
                .flatMapValues(List::stream)
                .mapValues(CpmPriceCampaign::getUid);

        Map<Integer, Set<Long>> uidsByShard = EntryStream.zip(agencyShards, agencyUids)
                .append(campaignUidsStream)
                .grouping(toSet());

        return EntryStream.of(uidsByShard)
                .flatMapKeyValue((shard, uids) -> userRepository.fetchByUids(shard, uids).stream())
                .toMap(User::getUid, identity());
    }

    private String extractMoneyAsText(WalletRestMoney walletRestMoney) {
        Money walletMoney = walletRestMoney.getRest().roundToCentDown();
        return MONEY_FORMATTER.format(walletMoney.bigDecimalValue()) + " " + walletMoney.getCurrencyCode();
    }

    private static DecimalFormat createMoneyFormatter() {
        DecimalFormat formatter = new DecimalFormat();
        formatter.setMaximumFractionDigits(2);
        formatter.setMinimumFractionDigits(2);
        formatter.setGroupingUsed(false);
        return formatter;
    }

    private Map<Long, String> getTargetingDefaultAdGroup(int shard, List<CpmPriceCampaign> campaignsInShard) {
        var campaignIds = StreamEx.of(campaignsInShard)
                .map(CpmPriceCampaign::getId)
                .toList();

        Map<Long, Long> campaignIdToPid = getCampaignIdToPid(shard, campaignIds);

        var retCondByAdGroupId = retConditionRepository.getRetConditionsByAdGroupIds(shard,
                campaignIdToPid.values());

        Map<Long, Goal> dbGoals = cryptaSegmentRepository.getByIds(
                StreamEx.of(retCondByAdGroupId.values())
                        .flatMap(List::stream)
                        .map(RetargetingCondition::collectGoals)
                        .flatMap(List::stream)
                        .map(Goal::getId)
                        .toSet()
        );

        return StreamEx.of(campaignIdToPid.keySet())
                .mapToEntry(identity(),
                        cid -> retConditionsAsText(retCondByAdGroupId.getOrDefault(
                                campaignIdToPid.get(cid), emptyList()),
                                dbGoals))
                .toMap();
    }

    private static String retConditionsAsText(List<RetargetingCondition> retCondList, Map<Long, Goal> dbGoals) {
        return StreamEx.of(retCondList)
                .map(RetargetingCondition::getRules)
                .flatMap(List::stream)
                .map(it -> ruleAsText(it, dbGoals))
                .joining("; ");
    }

    private static String ruleAsText(Rule rule, Map<Long, Goal> dbGoals) {
        return StreamEx.of(rule.getGoals())
                .map(goal -> goalName(goal, dbGoals))
                .joining(rule.getType() == RuleType.OR ? " или " : "и");
    }

    private static String goalName(Goal goal, Map<Long, Goal> dbGoals) {
        if (dbGoals.containsKey(goal.getId())) {
            return dbGoals.get(goal.getId()).getName();
        }
        return "Цель " + goal.getType() + " (" + goal.getId() + ")";
    }

    private Map<Long, String> getTargetingDefaultAdGroup(Map<Integer, List<CpmPriceCampaign>> campaignsByShard) {
        return EntryStream.of(campaignsByShard)
                .mapKeyValue((shard, campaignsInShard) ->
                        getTargetingDefaultAdGroup(shard, campaignsInShard))
                .flatMapToEntry(identity())
                .toMap();
    }

    private String fieldBrandlift(CpmPriceCampaign campaign) {
        return StringUtils.isEmpty(campaign.getBrandSurveyId()) ? "Нет" : "Да";
    }

    private String fieldAvgDayBudget(CpmPriceCampaign campaign, PricePackage pricePackage) {
        long days = campaign.getStartDate().until(nvl(campaign.getEndDate(), campaign.getStartDate()).plusDays(1), ChronoUnit.DAYS);
        BigDecimal avgDayBudget = pricePackage.getPrice()
                .multiply(BigDecimal.valueOf(campaign.getFlightOrderVolume()))
                .divide(BigDecimal.valueOf(1000), 2, RoundingMode.HALF_UP)
                .divide(BigDecimal.valueOf(days), 2, RoundingMode.HALF_UP);
        return MONEY_FORMATTER.format(avgDayBudget) + " " + pricePackage.getCurrency();
    }

    private GeoTree getPriceGeoTree() {
        return pricePackageService.getGeoTree();
    }
}
enum InventoriColor {
    GREEN("Можно утверждать"),
    RED("Нельзя утверждать"),
    ERROR("Ошибка проверки прогноза");

    private String text;

    InventoriColor(String text) {
        this.text = text;
    }

    public String getText() {
        return text;
    }
}
