package ru.yandex.direct.internaltools.tools.productrestrictions;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import org.assertj.core.util.diff.DiffUtils;
import org.assertj.core.util.diff.Patch;

import ru.yandex.direct.core.entity.adgroup.model.ProductRestrictionKey;
import ru.yandex.direct.core.entity.product.model.ProductRestriction;
import ru.yandex.direct.core.entity.product.model.ProductRestrictionCondition;
import ru.yandex.direct.core.entity.product.service.ProductService;
import ru.yandex.direct.utils.JsonUtils;

import static com.google.common.collect.Maps.uniqueIndex;

@ParametersAreNonnullByDefault
class ProductRestrictionDiffer {

    // не используем читалку JSON из JsonTools, т.к. там разрешается иметь в JSON неизвестные поля.
    private ObjectMapper strictJsonMapper = new ObjectMapper();

    /**
     * Рисует красивую строчку из уникального ключа ограничения продукта.
     * Полезно, если результирующие строчки также будут уникальными, иначе будет не понятно к чему относится дифф
     */
    static String renderProductRestrictionKey(ProductRestrictionKey key) {
        if (key.getCriterionType() == null) {
            return String.format("%s/%s", key.getAdGroupType(), key.getProductId());
        } else {
            return String.format("%s/%s/%s", key.getAdGroupType(), key.getCriterionType(), key.getProductId());
        }
    }

    /**
     * Рисует списки ограничений продукта в виде компактного JSON и возвращает строчку с диффом.
     * Если различий не найдено, возвращает пустую строку.
     * <p>
     * Функция умеет сравнивать объекты с незаполненным полем condition, в этом случае
     * условие десериализуется из поля conditionJson
     */
    String getProductRestrictionDiff(
            List<ProductRestriction> dbRestrictions,
            List<ProductRestriction> fileRestrictions
    ) {
        ImmutableMap<ProductRestrictionKey, ProductRestriction> oldProdRestrictions = uniqueIndex(
                dbRestrictions, ProductService::calculateUniqueProductRestrictionKey
        );
        ImmutableMap<ProductRestrictionKey, ProductRestriction> newProdRestrictions = uniqueIndex(
                fileRestrictions, ProductService::calculateUniqueProductRestrictionKey
        );

        ArrayList<ProductRestrictionKey> allKeysList = new ArrayList<>(
                Sets.union(newProdRestrictions.keySet(), oldProdRestrictions.keySet())
        );
        allKeysList.sort(ProductService.PRODUCT_RESTRICTION_KEY_COMPARATOR);

        StringBuilder patchBuilder = new StringBuilder();
        for (var key : allKeysList) {
            var oldPr = oldProdRestrictions.get(key);
            var newPr = newProdRestrictions.get(key);

            var oldJson = "";
            if (oldPr != null) {
                oldJson = productRestrictionToCompactJson(Collections.singletonList(oldPr));
            }

            var newJson = "";
            if (newPr != null) {
                if (newPr.getId() == null && oldPr != null) {
                    newPr.setId(oldPr.getId());
                }
                newJson = productRestrictionToCompactJson(Collections.singletonList(newPr));
            } else {
                patchBuilder.append(" !!! Удаление записей не поддерживается !!!\n");
            }

            if (newJson.equals(oldJson)) {
                continue;
            }

            if (newPr != null && oldPr != null && !newPr.getId().equals(oldPr.getId())) {
                patchBuilder.append(" !!! Запрещено менять ID рестрикшена !!!\n");
            }

            patchBuilder.append(String.format("Index: %s\n", renderProductRestrictionKey(key)));

            Patch<String> diff = DiffUtils.diff(
                    oldJson.lines().collect(Collectors.toList()),
                    newJson.lines().collect(Collectors.toList())
            );
            String patch = String.join("\n", DiffUtils.generateUnifiedDiff(
                    "PPCDICT.product_restrictions",
                    ProductService.PRODUCT_RESTRICTIONS_FILE,
                    oldJson.lines().collect(Collectors.toList()),
                    diff, 3
            ));
            patchBuilder.append(patch);
            patchBuilder.append('\n');
        }

        return patchBuilder.toString();
    }

    private String productRestrictionToCompactJson(Collection<ProductRestriction> productRestrictions) {
        StringBuilder sb = new StringBuilder();
        for (var pr : productRestrictions) {
            String conditionJson = null;
            List<ProductRestrictionCondition> conditions = pr.getConditions();
            if (conditions == null && pr.getConditionJson() != null) {
                try {
                    conditions = Arrays.asList(
                            strictJsonMapper.readValue(pr.getConditionJson(), ProductRestrictionCondition[].class)
                    );
                } catch (IOException e) {
                    throw new RuntimeException("Failed to parse condition JSON", e);
                }
            }
            if (conditions != null) {
                conditionJson = conditionsToJson(conditions, "  ");
            }

            sb.append(String.format(
                    "{\"id\": %s, \"adgroupType\": \"%s\", \"productId\": %s,\n" +
                            "  \"publicDescriptionKey\": \"%s\", \"publicNameKey\": \"%s\",\n" +
                            "  \"unitCountMin\": %s, \"unitCountMax\": %s,\n" +
                            "  \"condition\": %s,\n" +
                            "},\n",
                    pr.getId(), pr.getGroupType(), pr.getProductId(),
                    pr.getPublicDescriptionKey(), pr.getPublicNameKey(),
                    pr.getUnitCountMin(), pr.getUnitCountMax(),
                    conditionJson
            ));
        }
        return sb.toString();
    }

    private String conditionsToJson(List<ProductRestrictionCondition> conditions, String indent) {
        StringBuilder jsonBuilder = new StringBuilder();
        jsonBuilder.append("[\n");
        // conditions выводим по одному элементу на строку в виде канонизированного JSON
        // иначе они очень сильно растягиваются вертикально, и дифф не очень красиво рисуется
        for (var cond : conditions) {
            String condJson = JsonUtils.toDeterministicJson(cond);
            // trailing comma тоже для красоты диффа
            jsonBuilder.append(String.format("%s  %s,\n", indent, condJson));
        }
        jsonBuilder.append(indent);
        jsonBuilder.append("]");

        return jsonBuilder.toString();
    }
}
