package ru.yandex.direct.internaltools.tools.pushnotifications;

import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.eventlog.model.EventLogType;
import ru.yandex.direct.core.entity.eventlog.service.EventLogService;
import ru.yandex.direct.core.entity.usersnotifications.repository.UsersNotificationsRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.pushnotifications.model.SendInformationalPushParameters;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.internaltools.tools.pushnotifications.model.SendInformationalPushParameters.CLIENTID_FIELD_NAME;
import static ru.yandex.direct.internaltools.tools.pushnotifications.model.SendInformationalPushParameters.LINK_FIELD_NAME;
import static ru.yandex.direct.internaltools.tools.pushnotifications.model.SendInformationalPushParameters.MAX_TEXT_SIZE;
import static ru.yandex.direct.internaltools.tools.pushnotifications.model.SendInformationalPushParameters.TEXT_FIELD_NAME;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.getLongIdsFromString;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.isStringWithValidLongIds;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;

@Tool(
        name = "Добавка пуша со ссылкой клиентам, подписавшимся на информационное сообщение",
        label = "send_informational_push_tool",
        description = "Добавляет новое событие в EventLog для пуша всем клиентам на событие InformationalWithLink,\n" +
                "если не указаны конкретные ClientID",
        consumes = SendInformationalPushParameters.class,
        type = InternalToolType.WRITER
)
@Category(InternalToolCategory.OTHER)
@Action(InternalToolAction.ADD)
@AccessGroup({InternalToolAccessRole.SUPER})
@ParametersAreNonnullByDefault
public class SendInformationalPushTool implements BaseInternalTool<SendInformationalPushParameters> {
    private static final Logger logger = LoggerFactory.getLogger(SendInformationalPushTool.class);
    private static final String EVENT_NAME = EventLogType.CUSTOM_MESSAGE_WITH_LINK.name().toLowerCase();
    private final UsersNotificationsRepository usersNotificationRepositoty;
    private final ShardHelper shardHelper;
    private final EventLogService eventLogService;

    @Autowired
    public SendInformationalPushTool(UsersNotificationsRepository usersNotificationsRepository,
                                     ShardHelper shardHelper,
                                     EventLogService eventLogService) {
        this.usersNotificationRepositoty = usersNotificationsRepository;
        this.shardHelper = shardHelper;
        this.eventLogService = eventLogService;
    }

    @Override
    public ValidationResult<SendInformationalPushParameters, Defect> validate(SendInformationalPushParameters params) {
        ItemValidationBuilder<SendInformationalPushParameters, Defect> builder = ItemValidationBuilder.of(params);
        builder.item(params.getClientId(), CLIENTID_FIELD_NAME)
                .check(isStringWithValidLongIds(1));
        builder.item(params.getText(), TEXT_FIELD_NAME)
                .check(notBlank())
                .check(maxStringLength(MAX_TEXT_SIZE));
        builder.item(params.getUrl(), LINK_FIELD_NAME).check(validHref());
        return builder.getResult();
    }

    @Override
    public InternalToolResult process(SendInformationalPushParameters params) {
        if (params.getClientId() != null) {
            Set<Long> clientIds = getLongIdsFromString(params.getClientId());
            logger.info("Работаем со списком клиентов, клиентов {}", clientIds.size());
            eventLogService.addCustomMessageWithLink(clientIds, params.getUrl(), params.getText());
            return new InternalToolResult().withMessage("Добавлено " + clientIds.size() + " событие(-ий).");
        }
        if (params.getSendToEveryone() && params.getReallySendToEveryone()) {
            logger.info("Заказана отправка всем клиентам");
            AtomicInteger updatedRecords = new AtomicInteger(0);
            shardHelper.forEachShard(shard -> {
                List<Long> uids = usersNotificationRepositoty.getSubscribedUsers(shard, EVENT_NAME);
                List<Long> clientIds =
                        shardHelper.getClientIdsByUids(uids).values().stream()
                                .distinct().collect(Collectors.toList());
                eventLogService.addCustomMessageWithLink(clientIds, params.getUrl(), params.getText());
                updatedRecords.addAndGet(clientIds.size());
            });
            return new InternalToolResult().withMessage("Добавлено " + updatedRecords.toString() + " событие(-ий).");
        }
        logger.warn("Не указали список и галочек не проставили");
        return new InternalToolResult().withMessage("Ничего не сделано: ни списка не указано, ни галочки не " +
                "проставлены");
    }
}
