package ru.yandex.direct.internaltools.tools.redirects;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.banner.model.BannerWithHref;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.redirectcheckqueue.service.RedirectCheckQueueService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.redirects.container.RedirectCheckQueueItem;
import ru.yandex.direct.internaltools.tools.redirects.model.RedirectCheckQueueParameters;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.banner.service.BannerUtils.getValueIfAssignable;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.getLongIdsFromString;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.isStringWithValidLongIds;

@Tool(
        name = "Очередь простукивания на редирект",
        label = "redirect_check_queue",
        description = "Инструмент для добавления кампаний и баннеров в очередь на проверку\n"
                + "Отрицательные 'максимальные ожидания' - это запланированные на будущее проверки, все нормально, не баг.\n"
                + "Для добавления значения нужно задать список идентификаторов хотя бы в одном из полей ввода",
        consumes = RedirectCheckQueueParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.ADD)
@Category(InternalToolCategory.TESTING)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.SUPERREADER, InternalToolAccessRole.MANAGER,
        InternalToolAccessRole.SUPPORT})
@ParametersAreNonnullByDefault
public class RedirectCheckQueueTool extends MassInternalTool<RedirectCheckQueueParameters, RedirectCheckQueueItem> {
    private static final Logger logger = LoggerFactory.getLogger(RedirectCheckQueueTool.class);

    private final RedirectCheckQueueService redirectCheckQueueService;
    private final BannerService bannerService;

    @Autowired
    public RedirectCheckQueueTool(RedirectCheckQueueService redirectCheckQueueService,
                                  BannerService bannerService) {
        this.redirectCheckQueueService = redirectCheckQueueService;
        this.bannerService = bannerService;
    }

    @Override
    public ValidationResult<RedirectCheckQueueParameters, Defect> validate(
            RedirectCheckQueueParameters parameters) {
        ItemValidationBuilder<RedirectCheckQueueParameters, Defect> validationBuilder =
                ItemValidationBuilder.of(parameters);
        validationBuilder.check(Constraint.fromPredicate(oneOfItemsIsPresent(), CommonDefects.invalidValue()));

        validationBuilder.item(parameters.getCampaignIds(), "campaignIds")
                .check(isStringWithValidLongIds(1), When.notNull());
        validationBuilder.item(parameters.getBannerIds(), "bannerIds")
                .check(isStringWithValidLongIds(1), When.notNull());

        return validationBuilder.getResult();
    }

    private static Predicate<RedirectCheckQueueParameters> oneOfItemsIsPresent() {
        return s -> !((s.getBannerIds() == null || s.getBannerIds().isEmpty())
                && (s.getCampaignIds() == null || s.getCampaignIds().isEmpty()));
    }

    @Override
    protected List<RedirectCheckQueueItem> getMassData() {
        long now = LocalDateTime.now().toEpochSecond(ZoneOffset.UTC);
        return redirectCheckQueueService.getDomainCheckStat().stream()
                .map(s -> new RedirectCheckQueueItem()
                        .withDomain(s.getDomain())
                        .withCampaignsNum(s.getCampaignsNum())
                        .withBannersNum(s.getBannersNum())
                        .withSecondsInQueue(now - s.getOldestEntryAge().toEpochSecond(ZoneOffset.UTC)))
                .sorted(Comparator.comparing(RedirectCheckQueueItem::getSecondsInQueue, Comparator.reverseOrder()))
                .collect(Collectors.toList());
    }

    @Override
    protected List<RedirectCheckQueueItem> getMassData(RedirectCheckQueueParameters parameters) {
        Map<Long, String> bannerIdToLink = getBannersAndLinksToPush(parameters);
        if (!bannerIdToLink.isEmpty()) {
            pushBannersIntoQueue(parameters.getOperator(), bannerIdToLink);
        }
        return getMassData();
    }

    private Map<Long, String> getBannersAndLinksToPush(RedirectCheckQueueParameters parameters) {
        List<BannerWithSystemFields> banners = new ArrayList<>();
        if (parameters.getCampaignIds() != null) {
            Set<Long> campaignIds = getLongIdsFromString(parameters.getCampaignIds());
            banners.addAll(bannerService.getBannersByCampaignIds(campaignIds));
        }
        if (parameters.getBannerIds() != null) {
            Set<Long> bannerIds = getLongIdsFromString(parameters.getBannerIds());
            banners.addAll(bannerService.getBannersByIds(bannerIds));
        }
        return banners.stream()
                .collect(Collectors.toMap(BannerWithSystemFields::getId,
                        banner -> getValueIfAssignable(banner, BannerWithHref.HREF)));
    }

    private void pushBannersIntoQueue(User operator, Map<Long, String> bannerIdToLink) {
        logger.info("User {} is adding banners {} to redirect check queue", operator.getLogin(),
                bannerIdToLink.keySet());
        redirectCheckQueueService.removeLinksFromCache(bannerIdToLink.values());
        int num = redirectCheckQueueService.pushBannersIntoQueue(bannerIdToLink.keySet());
        logger.info("Successfully added {} entries to queue", num);
    }
}
