package ru.yandex.direct.internaltools.tools.requirefiltrationbydontshowdomains;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.requirefiltrationbydontshowdomains.model.CampaignTypesMode;
import ru.yandex.direct.internaltools.tools.requirefiltrationbydontshowdomains.model.SetRequireFiltrationByDontShowDomainsParams;
import ru.yandex.direct.internaltools.tools.requirefiltrationbydontshowdomains.model.SetRequireFiltrationByDontShowDomainsResult;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;

import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.campaign.model.CampaignType.CPM_BANNER;
import static ru.yandex.direct.core.entity.campaign.model.CampaignType.DYNAMIC;
import static ru.yandex.direct.core.entity.campaign.model.CampaignType.MOBILE_CONTENT;
import static ru.yandex.direct.core.entity.campaign.model.CampaignType.PERFORMANCE;
import static ru.yandex.direct.core.entity.campaign.model.CampaignType.TEXT;
import static ru.yandex.direct.internaltools.tools.requirefiltrationbydontshowdomains.SetRequireFiltrationByDontShowDomainsToolUtils.buildNewOpts;
import static ru.yandex.direct.internaltools.tools.requirefiltrationbydontshowdomains.model.CampaignTypesMode.ALL;
import static ru.yandex.direct.internaltools.tools.requirefiltrationbydontshowdomains.model.CampaignTypesMode.CPM_ONLY;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Tool(
        name = "Включить фильтрацию по списку запрещенных доменов на площадках Яндекса",
        label = "set_require_filtration_by_dont_show_domains",
        description = "Включение фильтрации по списку запрещенных доменов на площадках Яндекса.",
        consumes = SetRequireFiltrationByDontShowDomainsParams.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.SET)
@Category(InternalToolCategory.OTHER)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.MANAGER})
@ParametersAreNonnullByDefault
public class SetRequireFiltrationByDontShowDomainsTool extends MassInternalTool<SetRequireFiltrationByDontShowDomainsParams, SetRequireFiltrationByDontShowDomainsResult> {

    private static final int CHUNK_SIZE = 1000;

    private static final Map<CampaignTypesMode, Set<CampaignType>> CAMPAIGN_TYPES_BY_MODE = Map.of(
            CPM_ONLY, Set.of(CPM_BANNER),
            ALL, Set.of(TEXT, MOBILE_CONTENT, CPM_BANNER, DYNAMIC, PERFORMANCE));

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;

    @Autowired
    public SetRequireFiltrationByDontShowDomainsTool(ShardHelper shardHelper, CampaignRepository campaignRepository,
                                                     DslContextProvider dslContextProvider) {
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.dslContextProvider = dslContextProvider;
    }

    @Override
    protected List<SetRequireFiltrationByDontShowDomainsResult> getMassData(SetRequireFiltrationByDontShowDomainsParams parameter) {
        Set<CampaignType> campaignTypes = CAMPAIGN_TYPES_BY_MODE.get(parameter.getCampaignTypesMode());

        if (parameter.getLogins() != null) {
            return processByLogins(parameter.getLogins(), campaignTypes);
        } else if (parameter.getCampaignIds() != null) {
            return processByCampaignIds(parameter.getCampaignIds(), campaignTypes);
        } else {
            return emptyList();
        }
    }

    private List<SetRequireFiltrationByDontShowDomainsResult> processByLogins(String loginsRaw,
                                                                             Set<CampaignType> campaignTypes) {
        List<String> logins = StreamEx.of(loginsRaw.split(","))
                .map(String::trim)
                .toList();

        Set<ClientId> clientIds = EntryStream.of(shardHelper.getClientIdsByLogins(mapList(logins, identity())))
                .mapValues(ClientId::fromLong)
                .values()
                .toSet();

        Set<Long> campaignIds = shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID, ClientId::asLong).stream()
                .flatMapKeyValue((shard, clientIdsInShard) ->
                        StreamEx.of(campaignRepository.getCampaignIdsByClientIds(shard, clientIdsInShard)))
                .toSet();

        return processCampaigns(campaignIds, campaignTypes);
    }

    private List<SetRequireFiltrationByDontShowDomainsResult> processByCampaignIds(String campaignIdsRaw,
                                                                                  Set<CampaignType> campaignTypes) {
        Set<Long> campaignIds = StreamEx.of(campaignIdsRaw.split(","))
                .map(String::trim)
                .map(Long::parseLong)
                .toSet();

        return processCampaigns(campaignIds, campaignTypes);
    }

    private List<SetRequireFiltrationByDontShowDomainsResult> processCampaigns(Set<Long> campaignIds,
                                                                              Set<CampaignType> campaignTypes) {
        shardHelper.groupByShard(campaignIds, ShardKey.CID).chunkedBy(CHUNK_SIZE).forEach((shard, campaignIdsInShard) ->
                dslContextProvider.ppcTransaction(shard, configuration -> {
                    DSLContext dslContext = configuration.dsl();

                    List<Campaign> campaigns =
                            campaignRepository.selectCampaignsForUpdate(dslContext, campaignIds, campaignTypes);

                    List<AppliedChanges<Campaign>> appliedChanges = StreamEx.of(campaigns)
                            .map(campaign -> new ModelChanges<>(campaign.getId(), Campaign.class)
                                    .process(buildNewOpts(campaign.getOpts()), Campaign.OPTS)
                                    .applyTo(campaign))
                            .toList();

                    campaignRepository.updateCampaigns(dslContext, appliedChanges);
                })
        );

        return emptyList();
    }
}
