package ru.yandex.direct.internaltools.tools.searchspamusers;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.internaltools.tools.searchspamusers.container.SearchSpamUsersInfo;

import static org.jooq.impl.DSL.field;
import static org.jooq.impl.DSL.ifnull;
import static org.jooq.impl.DSL.sum;
import static org.jooq.impl.DSL.when;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;

@Repository
@ParametersAreNonnullByDefault
public class SearchSpamUsersRepository {
    private static final String DOMAIN_ALIAS = "domain";
    private static final String ACCEPTED_ALIAS = "cnt_accepted";
    private static final String DECLINED_ALIAS = "cnt_declined";
    private final DslContextProvider dslContextProvider;

    @Autowired
    public SearchSpamUsersRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    public List<SearchSpamUsersInfo> getSearchSpamUsersData(
            int shard,
            Collection<Long> uids) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.UID,
                        DSL.groupConcatDistinct(ifnull(BANNERS.DOMAIN, "")).as(DOMAIN_ALIAS),
                        sum(field(when(BANNERS.STATUS_MODERATE.eq(BannersStatusmoderate.No), 1).otherwise(0)))
                                .as(DECLINED_ALIAS),
                        sum(field(when(BANNERS.STATUS_MODERATE.eq(BannersStatusmoderate.Yes), 1).otherwise(0)))
                                .as(ACCEPTED_ALIAS)
                )
                .from(CAMPAIGNS)
                .leftJoin(BANNERS).on(BANNERS.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.UID.in(uids))
                .groupBy(CAMPAIGNS.UID)
                .fetch()
                .map(record -> new SearchSpamUsersInfo()
                        .withUid(record.get(CAMPAIGNS.UID))
                        .withDomain(record.get(DOMAIN_ALIAS, String.class).replaceAll(",", ", "))
                        .withDenied(record.get(DECLINED_ALIAS, Integer.class))
                        .withAccepted(record.get(ACCEPTED_ALIAS, Integer.class)));
    }
}
