package ru.yandex.direct.internaltools.tools.searchspamusers;

import java.time.temporal.ChronoUnit;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.searchspamusers.container.SearchSpamUsersInfo;
import ru.yandex.direct.internaltools.tools.searchspamusers.model.SearchSpamUsersParameters;
import ru.yandex.direct.internaltools.tools.searchspamusers.service.SearchSpamUsersService;
import ru.yandex.direct.internaltools.tools.searchspamusers.service.SearchSpamUsersValidationService;
import ru.yandex.direct.internaltools.utils.ToolParameterUtils;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.constraint.NumberConstraints;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

@Tool(
        name = "Поиск логинов по IP",
        label = "search_spam_users",
        description =
                "Если указаны логины, то находим какие еще пользователи заходили с этих же ip за указанный период.\n"
                        + "Если указан IP — список логинов по этому ip за указанный период.",
        consumes = SearchSpamUsersParameters.class,
        type = InternalToolType.REPORT
)
@Category(InternalToolCategory.OTHER)
@Action(InternalToolAction.SHOW)
@AccessGroup({InternalToolAccessRole.INTERNAL_USER})
@ParametersAreNonnullByDefault
public class SearchSpamUsersTool extends MassInternalTool<SearchSpamUsersParameters, SearchSpamUsersInfo> {
    private final SearchSpamUsersService searchSpamUsersService;

    @Autowired
    public SearchSpamUsersTool(SearchSpamUsersService searchSpamUsersService) {
        this.searchSpamUsersService = searchSpamUsersService;
    }

    @Override
    protected List<SearchSpamUsersInfo> getMassData(SearchSpamUsersParameters parameter) {
        return searchSpamUsersService
                .getSpamUsersInfo(ToolParameterUtils.parseCommaSeparatedString(parameter.getLogins()),
                        parameter.getIp(), parameter.getLogTable(), parameter.getDateFrom(), parameter.getDateTo());
    }

    @Override
    public ValidationResult<SearchSpamUsersParameters, Defect> validate(
            SearchSpamUsersParameters parameters) {
        ItemValidationBuilder<SearchSpamUsersParameters, Defect> validationBuilder =
                ItemValidationBuilder.of(parameters);

        validationBuilder
                .item(ChronoUnit.DAYS.between(parameters.getDateFrom(), parameters.getDateTo()), "date range")
                .check(NumberConstraints.inRange(0L, SearchSpamUsersValidationService.MAX_RANGE_IN_DAYS))
                .item(parameters, "logins or ip")
                .check(SearchSpamUsersValidationService.loginsOrIpPresentInParameters());
        return validationBuilder.getResult();
    }
}
