package ru.yandex.direct.internaltools.tools.searchspamusers.service;

import java.sql.Date;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.direct.clickhouse.SqlBuilder;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.dbutil.wrapper.SimpleDb;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.internaltools.tools.searchspamusers.SearchSpamUsersRepository;
import ru.yandex.direct.internaltools.tools.searchspamusers.container.SearchSpamUsersInfo;
import ru.yandex.direct.internaltools.tools.searchspamusers.model.SearchSpamUsersLogTableEnum;
import ru.yandex.direct.internaltools.tools.searchspamusers.model.UidIpPair;
import ru.yandex.direct.logviewercore.domain.LogRecordInfo;
import ru.yandex.direct.logviewercore.domain.LogRecordRowMapper;
import ru.yandex.direct.logviewercore.domain.LogTablesInfoManager;
import ru.yandex.direct.logviewercore.domain.ppclog.LogApiRecord;
import ru.yandex.direct.logviewercore.domain.ppclog.LogCmdRecord;
import ru.yandex.direct.logviewercore.domain.ppclog.LogRecord;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.utils.FunctionalUtils;
import ru.yandex.inside.passport.blackbox.Blackbox;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class SearchSpamUsersService {
    private static final String BLACKBOX_UID_FIELD = "accounts.login.uid";
    private static final String LOCALHOST_IP = "127.0.0.1";
    private static final String UID_FIELD = "uid";
    private static final String IP_FIELD = "ip";
    private static final String SERVICE_FIELD = "service";
    private static final List<String> CMD_LOG_SERVICES = ImmutableList.of("direct.perl.web",
            "direct.java.webapi",
            // пустая строка - для обратной совместимости со старыми логами, залитыми до ~2018-04-01
            "");
    private static final ImmutableList<String> FIELDS_TO_FETCH_AND_GROUP_BY = ImmutableList.of(UID_FIELD, IP_FIELD);

    private final ShardHelper shardHelper;
    private final DatabaseWrapperProvider dbProvider;
    private final SearchSpamUsersRepository searchSpamUsersRepository;
    private final UserService userService;
    private final String blackboxEndpoint;
    private final TvmIntegration tvmIntegration;
    private final TvmService tvmBlackboxService;

    @Autowired
    public SearchSpamUsersService(ShardHelper shardHelper,
                                  DatabaseWrapperProvider dbProvider,
                                  SearchSpamUsersRepository searchSpamUsersRepository,
                                  UserService userService,
                                  @Value("${blackbox.endpoint}") String blackboxEndpoint,
                                  TvmIntegration tvmIntegration,
                                  EnvironmentType environmentType) {
        this.shardHelper = shardHelper;
        this.dbProvider = dbProvider;
        this.searchSpamUsersRepository = searchSpamUsersRepository;
        this.userService = userService;
        this.blackboxEndpoint = blackboxEndpoint;
        this.tvmIntegration = tvmIntegration;
        this.tvmBlackboxService = environmentType.isProductionOrPrestable()
                ? TvmService.BLACKBOX_PROD
                : TvmService.BLACKBOX_MIMINO;
    }

    public List<SearchSpamUsersInfo> getSpamUsersInfo(Set<String> logins, String ip,
                                                      SearchSpamUsersLogTableEnum logTableEnum, LocalDate dateFrom,
                                                      LocalDate dateTo) {
        Set<String> ipList = new HashSet<>();
        Set<Long> uids = getUidsByLogins(logins);
        if (!uids.isEmpty()) {
            ipList = getIpsForUids(dateFrom, dateTo, uids, logTableEnum);
        }

        //Если не нашли посещений по uid'ам и не получили IP, то все. Если есть IP работаем с ним.
        if (ipList.isEmpty()) {
            if (ip.isEmpty()) {
                return Collections.emptyList();
            } else {
                ipList = ImmutableSet.of(ip);
            }
        }

        List<UidIpPair> uidIpsPairList = getNewUidsForIps(dateFrom, dateTo, uids, ipList, logTableEnum);

        Map<Long, User> usersMap = FunctionalUtils.listToMap(
                userService.massGetUser(uidIpsPairList.stream()
                        .map(UidIpPair::getUid)
                        .collect(Collectors.toList())),
                User::getUid);

        Set<Long> potentialSpamUids = uidIpsPairList.stream()
                .map(UidIpPair::getUid)
                .collect(Collectors.toSet());

        Map<Long, SearchSpamUsersInfo> fetchedData =
                FunctionalUtils.listToMap(shardHelper.groupByShard(potentialSpamUids, ShardKey.UID).stream()
                        .map(integerListEntry -> searchSpamUsersRepository
                                .getSearchSpamUsersData(integerListEntry.getKey(), integerListEntry.getValue()))
                        .flatMap(List::stream)
                        .collect(Collectors.toList()), SearchSpamUsersInfo::getUid);


        for (UidIpPair uidIpPair : uidIpsPairList) {
            if (fetchedData.containsKey(uidIpPair.getUid())) {
                User user = usersMap.get(uidIpPair.getUid());
                fetchedData.get(uidIpPair.getUid())
                        .withIp(uidIpPair.getIp())
                        .withLogin(user.getLogin())
                        .withFio(user.getLogin())
                        .withBlocked(user.getStatusBlocked());

            }
        }
        return new ArrayList<>(fetchedData.values());
    }

    /**
     * Метод для получения сета uid'ов по логинам через blackbox
     *
     * @param logins сет логинов, для получения uid
     * @return сет уникальных идентификаторов пользователя
     */
    private Set<Long> getUidsByLogins(Set<String> logins) {
        Blackbox blackbox = new Blackbox(blackboxEndpoint);
        String tvmTicket = tvmIntegration.getTicket(tvmBlackboxService);
        Set<Long> uids = new HashSet<>();
        for (String login : logins) {
            Optional<Long> uidOption = blackbox.userInfo(HttpUtil.getRemoteAddressForBlackbox(), login,
                    Collections.singletonList(BLACKBOX_UID_FIELD), tvmTicket).getUidLong().toOptional();
            uidOption.ifPresent(uids::add);
        }
        return uids;
    }

    /**
     * Метод для получения сета IP адресов, с которых были обращения в api или cmd
     * с указанных uids в указанный временной промежуток
     *
     * @param dateFrom     дата начиная с которой ищем IP
     * @param dateTo       дата до которой ищем IP
     * @param uids         сет uid'ов посещения которых интересуют
     * @param logTableEnum в каких логах ищем
     * @return сет IP
     */
    private Set<String> getIpsForUids(LocalDate dateFrom, LocalDate dateTo, Set<Long> uids,
                                      SearchSpamUsersLogTableEnum logTableEnum) {
        Set<String> ipList;

        List<? extends LogRecord> logRecords = getLogRowsForUids(logTableEnum, dateFrom, dateTo, uids);
        if (logTableEnum.equals(SearchSpamUsersLogTableEnum.API)) {
            ipList = logRecords.stream().map(o -> ((LogApiRecord) o).ip).collect(Collectors.toSet());
        } else {
            ipList = logRecords.stream().map(o -> ((LogCmdRecord) o).ip).collect(Collectors.toSet());
        }

        if (ipList == null || ipList.isEmpty()) {
            return Collections.emptySet();
        }

        return ipList;
    }

    /**
     * Метод для получения списка пар uid—IP адресов,
     *
     * @param dateFrom     дата начиная с которой ищем
     * @param dateTo       дата до которой ищем
     * @param uids         список uid'ов для которых уже известны посещения
     * @param ipList       списко IP для которых ищем
     * @param logTableEnum в каких логах ищем
     * @return лист пар uid-ip
     */
    private List<UidIpPair> getNewUidsForIps(LocalDate dateFrom, LocalDate dateTo,
                                             Set<Long> uids, Set<String> ipList, SearchSpamUsersLogTableEnum logTableEnum) {
        List<? extends LogRecord> logRecords = getLogRowsForIps(logTableEnum, dateFrom, dateTo, uids, ipList);

        List<UidIpPair> result;
        if (logTableEnum.equals(SearchSpamUsersLogTableEnum.API)) {
            result = logRecords.stream().map(o -> new UidIpPair(((LogApiRecord) o).uid, ((LogApiRecord) o).ip))
                    .collect(Collectors.toList());

        } else {
            result = logRecords.stream().map(o -> new UidIpPair(((LogCmdRecord) o).uid, ((LogCmdRecord) o).ip))
                    .collect(Collectors.toList());
        }

        if (result.isEmpty()) {
            return result;
        }
        return result;
    }


    private List<? extends LogRecord> getLogRowsForUids(
            SearchSpamUsersLogTableEnum logTableEnum,
            LocalDate dateFrom, LocalDate dateTo,
            Set<Long> uids) {

        LogRecordInfo<? extends LogRecord> info = LogTablesInfoManager.getLogRecordInfo(logTableEnum.getTableName());
        LogRecordRowMapper<? extends LogRecord> mapper = new LogRecordRowMapper<>(FIELDS_TO_FETCH_AND_GROUP_BY, info);

        SqlBuilder sql = new SqlBuilder()
                .from(info.getTableName())
                .select(mapList(FIELDS_TO_FETCH_AND_GROUP_BY, info::getDbColumn));

        if (logTableEnum == SearchSpamUsersLogTableEnum.CMD) {
            sql.whereIn(SqlBuilder.column(SERVICE_FIELD), CMD_LOG_SERVICES);
        }
        sql.where(SqlBuilder.column(info.getLogDateColumn()), ">=", Date.valueOf(dateFrom));
        sql.where(SqlBuilder.column(info.getLogDateColumn()), "<=", Date.valueOf(dateTo));
        sql.whereIn(SqlBuilder.column(UID_FIELD), uids);
        sql.groupBy(FIELDS_TO_FETCH_AND_GROUP_BY.toArray(new String[]{}));

        return dbProvider.get(SimpleDb.CLICKHOUSE_CLOUD).query(sql.toString(), sql.getBindings(), mapper);
    }

    private List<? extends LogRecord> getLogRowsForIps(
            SearchSpamUsersLogTableEnum logTableEnum, LocalDate dateFrom, LocalDate dateTo,
            Set<Long> uids, Set<String> ipList) {
        LogRecordInfo<? extends LogRecord> info = LogTablesInfoManager.getLogRecordInfo(logTableEnum.getTableName());
        LogRecordRowMapper<? extends LogRecord> mapper = new LogRecordRowMapper<>(FIELDS_TO_FETCH_AND_GROUP_BY, info);

        SqlBuilder sql = new SqlBuilder()
                .from(info.getTableName())
                .select(mapList(FIELDS_TO_FETCH_AND_GROUP_BY, info::getDbColumn));

        if (logTableEnum == SearchSpamUsersLogTableEnum.CMD) {
            sql.whereIn(SqlBuilder.column(SERVICE_FIELD), CMD_LOG_SERVICES);
        }
        sql.where(SqlBuilder.column(info.getLogDateColumn()), ">=", Date.valueOf(dateFrom));
        sql.where(SqlBuilder.column(info.getLogDateColumn()), "<=", Date.valueOf(dateTo));
        if (!uids.isEmpty()) {
            sql.whereNotIn(SqlBuilder.column(UID_FIELD), uids);
        }
        sql.whereIn(SqlBuilder.column(IP_FIELD), ipList);
        sql.groupBy(FIELDS_TO_FETCH_AND_GROUP_BY.toArray(new String[]{}));

        return dbProvider.get(SimpleDb.CLICKHOUSE_CLOUD).query(sql.toString(), sql.getBindings(), mapper);
    }
}
