package ru.yandex.direct.internaltools.tools.social;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.feature.container.ChiefRepresentativeWithClientFeature;
import ru.yandex.direct.core.entity.feature.model.FeatureState;
import ru.yandex.direct.core.entity.feature.service.FeatureCache;
import ru.yandex.direct.core.entity.feature.service.FeatureManagingService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.social.models.InternalToolsSocialClientInfo;
import ru.yandex.direct.internaltools.tools.social.models.InternalToolsSocialFeatureIdsState;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.internaltools.tools.feature.container.InternalToolsFeatureConverter.fromFeatureState;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Tool(
        name = "Список клиентов с фичой социальной рекламы",
        label = "get_social_clients_clients_for_features",
        description = "Получение списка клиентов с включенным доступом",
        consumes = InternalToolsSocialFeatureIdsState.class,
        type = InternalToolType.REPORT
)
@Action(InternalToolAction.SHOW)
@Category(InternalToolCategory.SOCIAL)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPPORT,
        InternalToolAccessRole.PLACER})
public class GetSocialClientsTool extends MassInternalTool<InternalToolsSocialFeatureIdsState,
        InternalToolsSocialClientInfo> {
    @Autowired
    private FeatureManagingService clientFeatureService;

    @Autowired
    private FeatureCache featureCache;

    public InternalToolsSocialClientInfo toInternalToolsSocialClientInfo(
            ChiefRepresentativeWithClientFeature chiefRepresentativeWithClientFeature,
            Map<ClientId, FeatureState> payableStates
    ) {

        return new InternalToolsSocialClientInfo()
                .withClientId(chiefRepresentativeWithClientFeature.getClientFeature().getClientId().asLong())
                .withIsEnabled(fromFeatureState(
                        payableStates.get(chiefRepresentativeWithClientFeature.getClientFeature().getClientId())
                ))
                .withChiefUid(chiefRepresentativeWithClientFeature.getChiefUid())
                .withChiefLogin(chiefRepresentativeWithClientFeature.getChiefLogin());

    }

    @Override
    protected List<InternalToolsSocialClientInfo> getMassData(InternalToolsSocialFeatureIdsState parameter) {

        Result<Map<Long, List<ChiefRepresentativeWithClientFeature>>> operationResult =
                clientFeatureService.getFeaturesClients(
                        List.of(FeatureName.SOCIAL_ADVERTISING.getName().toLowerCase(),
                                FeatureName.SOCIAL_ADVERTISING_PAYABLE.getName().toLowerCase()
                        ),
                        FeatureState.ENABLED);

        if (hasValidationIssues(operationResult)) {
            throw new InternalToolValidationException("").withValidationResult(operationResult.getValidationResult());
        }

        var featureNamesToIds =
                featureCache.getIdsByTextId(List.of(FeatureName.SOCIAL_ADVERTISING.getName().toLowerCase(),
                        FeatureName.SOCIAL_ADVERTISING_PAYABLE.getName().toLowerCase()));

        var socialAdvertisingId = featureNamesToIds.get(FeatureName.SOCIAL_ADVERTISING.getName().toLowerCase());
        var socialAdvertisingPayableId =
                featureNamesToIds.get(FeatureName.SOCIAL_ADVERTISING_PAYABLE.getName().toLowerCase());

        @SuppressWarnings("OptionalGetWithoutIsPresent")
        List<ChiefRepresentativeWithClientFeature> chiefRepresentativeWithClientFeatures =
                operationResult.getResult().getOrDefault(socialAdvertisingId, List.of());

        Map<ClientId, FeatureState> result =
                operationResult.getResult().getOrDefault(socialAdvertisingPayableId, List.of())
                        .stream()
                        .collect(Collectors.toMap(e -> e.getClientFeature().getClientId(),
                                e -> e.getClientFeature().getState()));

        return mapList(chiefRepresentativeWithClientFeatures, e -> toInternalToolsSocialClientInfo(e, result));
    }

    @Override
    public ValidationResult<InternalToolsSocialFeatureIdsState, Defect> validate(
            InternalToolsSocialFeatureIdsState internalToolsFeatureIdsState) {

        ItemValidationBuilder<InternalToolsSocialFeatureIdsState, Defect> validationBuilder =
                ItemValidationBuilder.of(internalToolsFeatureIdsState);

        return validationBuilder.getResult();
    }
}
