package ru.yandex.direct.internaltools.tools.social;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.container.FeatureTextIdToClientIdState;
import ru.yandex.direct.core.entity.feature.container.LoginClientIdChiefLoginWithState;
import ru.yandex.direct.core.entity.feature.model.FeatureState;
import ru.yandex.direct.core.entity.feature.service.FeatureManagingService;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.social.models.InternalToolsSocialAdvertisingFeature;
import ru.yandex.direct.internaltools.tools.social.models.InternalToolsSocialFeatureResponse;
import ru.yandex.direct.internaltools.utils.ToolParameterUtils;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.matchPattern;

@Tool(
        name = "Подключение социальной рекламы",
        label = "enable_social_advertising_feature",
        description =
                "Включение и выключение социальной рекламы клиентам по списку логинов" +
                        " (в базе опция добавляется на ClientID, доступ будет у всех представителей)." +
                        " Если действие успешно, на этой странице появится таблица с данными клиента.\n" +
                        "Для отключения рекламы – отключите чекбокс «подключить».",
        consumes = InternalToolsSocialAdvertisingFeature.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.SET)
@Category(InternalToolCategory.SOCIAL)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.SUPPORT,
        InternalToolAccessRole.PLACER})
@ParametersAreNonnullByDefault
public class SocialClientsEnableByLoginsTool extends MassInternalTool<InternalToolsSocialAdvertisingFeature,
        InternalToolsSocialFeatureResponse> {

    public static final String LOGIN = "[^,]*+";
    public static final String LOGINS_SEPARATED_BY_COMMAS_PATTERN = LOGIN + "(," + LOGIN + ")*+";
    private static final Logger logger = LoggerFactory.getLogger(SocialClientsEnableByLoginsTool.class);
    private static final List<FeatureName> FEATURES_TO_TURN_OFF =
            List.of(FeatureName.ENABLE_UC_DNA_USER_CHOICE,
                    FeatureName.ENABLE_UC_DNA_USER_CHOICE_SUBSCRIPTION_TAB,
                    FeatureName.CAMPAIGN_SOURCE_USLUGI_CREATABLE);


    @Autowired
    private FeatureManagingService clientFeatureService;

    @Autowired
    private PpcPropertiesSupport ppcPropertiesSupport;

    @Autowired
    private ClientService clientService;

    @Autowired
    private CampaignService campaignService;

    @Autowired
    private PricePackageService pricePackageService;

    @Autowired
    private UserService userService;

    @Override
    public ValidationResult<InternalToolsSocialAdvertisingFeature, Defect> validate(
            InternalToolsSocialAdvertisingFeature internalToolsFeatureIdClientLoginsState) {

        ItemValidationBuilder<InternalToolsSocialAdvertisingFeature, Defect> validationBuilder =
                ItemValidationBuilder.of(internalToolsFeatureIdClientLoginsState);

        validationBuilder.check(notNull());

        validationBuilder.item(internalToolsFeatureIdClientLoginsState.getClientLogins().replaceAll("\\s+", ""),
                InternalToolsSocialAdvertisingFeature.LOGINS)
                .check(notNull())
                .check(matchPattern(LOGINS_SEPARATED_BY_COMMAS_PATTERN), When.isValidAnd(When.notNull()));

        return validationBuilder.getResult();
    }

    List<FeatureTextIdToClientIdState> toFeatureTextIdToClientIdStateList(
            FeatureName featureName, Collection<ClientId> clientIds, boolean enable) {
        return StreamEx.of(clientIds)
                .map(
                        clientId -> new FeatureTextIdToClientIdState()
                                .withTextId(featureName.getName())
                                .withState(enable ? FeatureState.ENABLED : FeatureState.DISABLED)
                                .withClientId(clientId)
                ).toList();
    }

    void turnOffFeatures(Set<ClientId> clientIds) {
        for (var i: FEATURES_TO_TURN_OFF) {
            var result =
                    clientFeatureService.switchFeaturesStateForClientIds(
                            toFeatureTextIdToClientIdStateList(
                                    i, clientIds, false));
            if (hasValidationIssues(result)) {
                throw new InternalToolValidationException("Ошибка при изменении фичи " +  i.getName())
                        .withValidationResult(result.getValidationResult());
            }
        }
    }

    @Override
    protected List<InternalToolsSocialFeatureResponse> getMassData(InternalToolsSocialAdvertisingFeature parameter) {

        var packages = getSocialPackages();

        var logins = ToolParameterUtils.parseLogins(parameter.getClientLogins());
        var loginToClientID = userService.massGetClientIdsByLogin(List.copyOf(logins));

        checkClientsHaveNoCampaigns(loginToClientID);

        var clientIds = listToSet(loginToClientID.values(), ClientId::fromLong);

        var enableSocialAdv = parameter.getIsAdvEnabled();

        var switchSocialAdvResult =
                clientFeatureService.switchFeaturesStateForClientIds(
                        toFeatureTextIdToClientIdStateList(
                                FeatureName.SOCIAL_ADVERTISING, clientIds, enableSocialAdv));

        clientService.updateSocialAdvertisingByClientIds(clientIds, enableSocialAdv);

        var allowPay = parameter.getIsPaymentAllowed();

        var switchSocialPayResult =
                clientFeatureService.switchFeaturesStateForClientIds(
                        toFeatureTextIdToClientIdStateList(
                                FeatureName.SOCIAL_ADVERTISING_PAYABLE, clientIds, allowPay));

        if (hasValidationIssues(switchSocialAdvResult)) {
            throw new InternalToolValidationException("Ошибка при изменении фичи SOCIAL_ADVERTISING")
                    .withValidationResult(switchSocialAdvResult.getValidationResult());
        }
        if (hasValidationIssues(switchSocialPayResult)) {
            throw new InternalToolValidationException("Ошибка при изменении фичи SOCIAL_ADVERTISING_PAYABLE")
                    .withValidationResult(switchSocialPayResult.getValidationResult());
        }

        if (enableSocialAdv) {
            turnOffFeatures(clientIds);
        }

        var clientIdToPackageIds = StreamEx.of(clientIds).toMap(clientId -> listToSet(packages));

        if (enableSocialAdv) {
            pricePackageService.addPackagesToClients(clientIdToPackageIds);
        } else {
            pricePackageService.deletePackagesFromClients(clientIdToPackageIds);
        }

        var response =
                toResponse(loginToClientID, switchSocialAdvResult, switchSocialPayResult);

        logger.info("{} social feature {}", enableSocialAdv ? "Adding" : "Removing", response);

        return response;
    }

    private List<InternalToolsSocialFeatureResponse> toResponse(
            Map<String, Long> loginToClientId,
            Result<List<LoginClientIdChiefLoginWithState>> switchSocialAdvResult,
            Result<List<LoginClientIdChiefLoginWithState>> switchSocialPayResult
    ) {
        var clientIdToClientWithStateSocialAdv =
                listToMap(switchSocialAdvResult.getResult(), LoginClientIdChiefLoginWithState::getClientId);
        var clientIdToClientWithStateSocialPay =
                listToMap(switchSocialPayResult.getResult(), LoginClientIdChiefLoginWithState::getClientId);
        return EntryStream.of(loginToClientId)
                .mapKeyValue((login, clientTd) -> new InternalToolsSocialFeatureResponse()
                        .withLogin(login)
                        .withClientId(clientTd)
                        .withChiefLogin(clientIdToClientWithStateSocialAdv
                                .get(clientTd).getChiefLogin())
                        .withAdvEnabled(clientIdToClientWithStateSocialAdv
                                .get(clientTd).getFeatureState() == FeatureState.ENABLED)
                        .withPaymentAllowed(clientIdToClientWithStateSocialPay
                                .get(clientTd).getFeatureState() == FeatureState.ENABLED)
                )
                .toList();
    }

    private List<Long> getSocialPackages() {
        var packages = ppcPropertiesSupport.get(PpcPropertyNames.SOCIAL_ADVERTISING_PACKAGES).getOrDefault(emptyList());
        if (packages.size() != pricePackageService.getPricePackages(packages).size()) {
            throw new IllegalArgumentException("Невалидный список пакетов в проперти SOCIAL_ADVERTISING_PACKAGES");
        }
        return packages;
    }

    private void checkClientsHaveNoCampaigns(Map<String, Long> loginToClientId) {
        var campaignsByClientId =
                campaignService.searchAllCampaignsByClientIdsAndTypes(loginToClientId.values(), CampaignTypeKinds.ALL);
        var clientsWithNonWalletCampaigns = EntryStream.of(campaignsByClientId)
                .filterValues(campaigns -> StreamEx.of(campaigns).anyMatch(c -> c.getType() != CampaignType.WALLET))
                .keys()
                .toSet();
        if (!clientsWithNonWalletCampaigns.isEmpty()) {
            var clientsInfo = EntryStream.of(loginToClientId)
                    .filterValues(clientsWithNonWalletCampaigns::contains)
                    .mapKeyValue((login, clientID) -> login + " (ClientID = " + clientID + ")")
                    .toList();
            throw new InternalToolValidationException("Менять фичу социальности можно только клиентам без кампаний, " +
                    "эти клиенты имеют кампании: " + String.join(", ", clientsInfo));
        }
    }
}
