package ru.yandex.direct.internaltools.tools.takeout;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.takeout.service.TakeoutService;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.takeout.model.RequestTakeoutParameters;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Tool(
        name = "Добавление задания для выгрузки в takeout",
        label = "request_takeout",
        description = "Добавляет в dbqueue задание на выгрузку данных в takeout.\n"
                + "данне кладутся в шард клиента либо рандомно, если данного uid нет в базе\n"
                + "select *,uncompress(args) from dbqueue_jobs where job_type_id = 24",
        consumes = RequestTakeoutParameters.class,
        type = InternalToolType.WRITER
)
@Category(InternalToolCategory.TAKEOUT)
@Action(InternalToolAction.ADD)
@AccessGroup({InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPER})
@ParametersAreNonnullByDefault
public class RequestTakeoutTool implements BaseInternalTool<RequestTakeoutParameters> {
    private static final Logger logger = LoggerFactory.getLogger(RequestTakeoutTool.class);

    private final TakeoutService takeoutService;

    @Autowired
    public RequestTakeoutTool(TakeoutService takeoutService) {
        this.takeoutService = takeoutService;
    }

    @Override
    public ValidationResult<RequestTakeoutParameters, Defect> validate(RequestTakeoutParameters parameters) {
        ItemValidationBuilder<RequestTakeoutParameters, Defect> vb =
                ItemValidationBuilder.of(parameters);
        vb.item(parameters.getUid(), RequestTakeoutParameters.UID)
                .check(validId());
        vb.item(parameters.getJobId(), RequestTakeoutParameters.JOB_ID)
                .check(notBlank());
        return vb.getResult();
    }

    @Override
    public InternalToolResult process(RequestTakeoutParameters parameters) {
        takeoutService.requestData(parameters.getUid(), parameters.getJobId(), null);
        String info = String.format("takeout request added to dbqueue for uid %s job_id %s",
                parameters.getUid(), parameters.getJobId());
        logger.info(info);
        return new InternalToolResult().withMessage(info);
    }
}
