package ru.yandex.direct.internaltools.tools.takeout;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.takeout.model.TakeoutJobParams;
import ru.yandex.direct.core.entity.takeout.model.TakeoutJobResult;
import ru.yandex.direct.dbqueue.model.DbQueueJob;
import ru.yandex.direct.dbqueue.repository.DbQueueRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.takeout.model.UnarchiveJobParameters;

import static ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes.TAKEOUT_REQUEST;

@Tool(
        name = "Разархивирование джоб takeout",
        label = "unarchive_dbqueue_jobs",
        description = "Ставит в очередь задание из архива\n"
                + "select *,uncompress(args) from dbqueue_job_archive where job_type_id = 24",
        consumes = UnarchiveJobParameters.class,
        type = InternalToolType.WRITER
)
@Category(InternalToolCategory.TAKEOUT)
@Action(InternalToolAction.ADD)
@AccessGroup({InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPER})
@ParametersAreNonnullByDefault
public class UnarchiveJobTool implements BaseInternalTool<UnarchiveJobParameters> {
    private final DbQueueRepository dbQueueRepository;
    private final ShardHelper shardHelper;

    @Autowired
    public UnarchiveJobTool(DbQueueRepository dbQueueRepository, ShardHelper shardHelper) {
        this.dbQueueRepository = dbQueueRepository;
        this.shardHelper = shardHelper;
    }

    @Override
    public InternalToolResult process(UnarchiveJobParameters parameters) {
        Set<Long> jobIds = parameters.getJobIdsSet();
        Map<Integer, List<DbQueueJob<TakeoutJobParams, TakeoutJobResult>>> jobsMap = new HashMap<>();
        shardHelper.forEachShard(
                shard -> jobsMap.put(shard, dbQueueRepository.findArchivedJobsByIds(shard, TAKEOUT_REQUEST, jobIds))
        );

        StringBuilder m = new StringBuilder();

        for (var jobEntry : jobsMap.entrySet()) {
            var jobs = dbQueueRepository.insertJobs(jobEntry.getKey(), TAKEOUT_REQUEST, jobEntry.getValue());
            m.append("shard ").append(jobEntry.getKey()).append(":");
            m.append(StreamEx.of(jobs).map(DbQueueJob::getId).joining(","));
            m.append("\n");
        }

        return new InternalToolResult().withMessage("job inserted:\n " + m);
    }
}
