package ru.yandex.direct.internaltools.tools.telephony;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.net.URLConnection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.telephony.model.IntToolTelephonyPlayback;
import ru.yandex.direct.internaltools.tools.telephony.model.TelephonySetPlaybackParameters;
import ru.yandex.direct.telephony.client.TelephonyClient;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

@Tool(
        name = "Загрузить джингл Яндекса для Телефонии",
        label = "upload_playback",
        description = "Загрузить джингл в Телефонию и установить его для всех телефонов клиента.\n" +
                "Чтобы узнать текущий идентификатор джингла, оставьте поле 'Playback' пустым.",
        consumes = TelephonySetPlaybackParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.SET)
@Category(InternalToolCategory.CALLTRACKING)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class TelephonySetPlaybackTool
        extends MassInternalTool<TelephonySetPlaybackParameters, IntToolTelephonyPlayback> {

    private static final String ALLOWED_FILE_FORMAT = "audio/x-wav";

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final TelephonyClient telephonyClient;

    public TelephonySetPlaybackTool(PpcPropertiesSupport ppcPropertiesSupport, TelephonyClient telephonyClient) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.telephonyClient = telephonyClient;
    }

    @Override
    public ValidationResult<TelephonySetPlaybackParameters, Defect> validate(TelephonySetPlaybackParameters params) {
        var vb = ItemValidationBuilder.of(params, Defect.class);
        vb.item(params.getPlayback(), "Playback")
                .check(fromPredicate(p -> p.length > 0, CommonDefects.invalidValue()))
                .check(fromPredicate(this::isValidContentType, CommonDefects.invalidFormat()));
        return vb.getResult();
    }

    @Override
    protected List<IntToolTelephonyPlayback> getMassData(TelephonySetPlaybackParameters parameters) {
        String playbackId;
        if (parameters.getPlayback() == null) {
            playbackId = ppcPropertiesSupport.get(PpcPropertyNames.TELEPHONY_PLAYBACK_ID).getOrDefault("");
        } else {
            playbackId = telephonyClient.uploadPlayback(parameters.getPlayback()).getPlaybackId();
            ppcPropertiesSupport.set(PpcPropertyNames.TELEPHONY_PLAYBACK_ID.getName(), playbackId);
        }
        IntToolTelephonyPlayback result = new IntToolTelephonyPlayback().withPlaybackId(playbackId);
        return List.of(result);
    }

    private boolean isValidContentType(byte[] playback) {
        try {
            String contentType = URLConnection.guessContentTypeFromStream(new ByteArrayInputStream(playback));
            if (!ALLOWED_FILE_FORMAT.equals(contentType)) {
                return false;
            }
        } catch (IOException e) {
            return false;
        }
        return true;
    }
}
