package ru.yandex.direct.internaltools.tools.telephony;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneService;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneMapping;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.core.entity.trackingphone.model.PhoneNumber;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.telephony.model.IntToolTelephonyUpdate;
import ru.yandex.direct.internaltools.tools.telephony.model.TelephonyUpdateParameters;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.parseCommaSeparatedString;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Tool(
        name = "Обновить выданные номера",
        label = "update_telephony",
        description = "Отчет ищет телефоны с указанным номером Телефонии и запрашивает новый номер в Телефонии.",
        consumes = TelephonyUpdateParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.SET)
@Category(InternalToolCategory.CALLTRACKING)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class TelephonyUpdateTool extends MassInternalTool<TelephonyUpdateParameters, IntToolTelephonyUpdate> {

    private final TelephonyPhoneService telephonyPhoneService;
    private final ClientPhoneRepository clientPhoneRepository;

    public TelephonyUpdateTool(
            TelephonyPhoneService telephonyPhoneService,
            ClientPhoneRepository clientPhoneRepository
    ) {
        this.telephonyPhoneService = telephonyPhoneService;
        this.clientPhoneRepository = clientPhoneRepository;
    }

    @Override
    public ValidationResult<TelephonyUpdateParameters, Defect> validate(TelephonyUpdateParameters params) {
        var vb = ItemValidationBuilder.of(params, Defect.class);
        vb.item(params.getUpdateNumbers(), "updateNumbers")
                .check(notNull());
        return vb.getResult();
    }

    @Override
    protected List<IntToolTelephonyUpdate> getMassData(TelephonyUpdateParameters parameters) {
        List<String> numbersForUpdate = mapList(
                parseCommaSeparatedString(parameters.getUpdateNumbers()),
                p -> ClientPhoneMapping.phoneNumberToDb(new PhoneNumber().withPhone(p))
        );
        List<String> notUpdated = new ArrayList<>(numbersForUpdate);
        List<ClientPhone> existedPhones = clientPhoneRepository.getByTelephonyNumbers(numbersForUpdate);
        Set<ClientPhone> updatedPhones = telephonyPhoneService.reassignTelephonyPhones(existedPhones);
        List<String> updatedNumbers = mapList(
                updatedPhones,
                p -> ClientPhoneMapping.phoneNumberToDb(p.getTelephonyPhone())
        );
        notUpdated.removeAll(updatedNumbers);
        return mapList(notUpdated, this::convert);
    }

    private IntToolTelephonyUpdate convert(String oldPhone) {
        return new IntToolTelephonyUpdate()
                .withTelephonyPhone(oldPhone);
    }

}
