package ru.yandex.direct.internaltools.tools.telephony;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneService;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.telephony.model.UnlinkUnusedTelephonyParameters;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.getLongIdsFromString;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Tool(
        name = "Отвязать неиспользуемые номера Телефонии",
        label = "unlink_unused_telephony",
        description = "Отчет проверяет, что введенные id соответствуют номеру Телефонии и удаляет их, если они не привязаны к объявлениям",
        consumes = UnlinkUnusedTelephonyParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.SET)
@Category(InternalToolCategory.CALLTRACKING)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class UnlinkUnusedTelephonyTool implements BaseInternalTool<UnlinkUnusedTelephonyParameters> {
    private static final Logger logger = LoggerFactory.getLogger(UnlinkUnusedTelephonyTool.class);

    private final ClientPhoneRepository clientPhoneRepository;
    private final TelephonyPhoneService telephonyPhoneService;

    public UnlinkUnusedTelephonyTool(
            ClientPhoneRepository clientPhoneRepository,
            TelephonyPhoneService telephonyPhoneService) {
        this.clientPhoneRepository = clientPhoneRepository;
        this.telephonyPhoneService = telephonyPhoneService;
    }

    @Override
    public ValidationResult<UnlinkUnusedTelephonyParameters, Defect> validate(UnlinkUnusedTelephonyParameters params) {
        var vb = ItemValidationBuilder.of(params, Defect.class);
        vb.item(params.getClientPhoneIds(), "clientPhoneIds")
                .check(notNull());
        return vb.getResult();
    }

    @Override
    public InternalToolResult process(UnlinkUnusedTelephonyParameters params) {
        Set<Long> clientPhoneIds = getLongIdsFromString(params.getClientPhoneIds());
        List<ClientPhone> telephonyPhones = clientPhoneRepository.getTelephony(clientPhoneIds);
        Map<Long, String> telephonyServiceIds = listToMap(telephonyPhones, ClientPhone::getId, ClientPhone::getTelephonyServiceId);
        Map<ClientId, List<ClientPhone>> phonesByClientId = StreamEx.of(telephonyPhones)
                .groupingBy(ClientPhone::getClientId);
        for (ClientId clientId : phonesByClientId.keySet()) {
            List<Long> phoneIds = mapList(phonesByClientId.get(clientId), ClientPhone::getId);
            logger.info("Try to unlink phones for clientId {}: {}", clientId, phoneIds);
            Set<Long> bannerPhoneIds = clientPhoneRepository.getBannerIdsByPhoneId(clientId, phoneIds).keySet();
            Set<Long> campaignPhoneIds = clientPhoneRepository.getCampaignIdsByPhoneId(clientId, phoneIds).keySet();
            List<Long> phoneIdsToDelete = new ArrayList<>();
            for (Long phoneId : phoneIds) {
                if (bannerPhoneIds.contains(phoneId) || campaignPhoneIds.contains(phoneId)) {
                    continue;
                }
                phoneIdsToDelete.add(phoneId);
            }
            deleteAndUnlink(clientId, phoneIdsToDelete, telephonyServiceIds);
        }
        return new InternalToolResult().withMessage("Done");
    }

    private void deleteAndUnlink(ClientId clientId, List<Long> phoneIdsToDelete, Map<Long, String> serviceIds) {
        logger.info("PhoneIds to delete: {}", phoneIdsToDelete);
        clientPhoneRepository.delete(clientId, phoneIdsToDelete);
        for (Long id : phoneIdsToDelete) {
            String serviceNumberId = serviceIds.get(id);
            logger.info("TelephonyServiceId to unlink: {}", serviceNumberId);
            telephonyPhoneService.detachTelephony(serviceNumberId);
        }
    }

}
