package ru.yandex.direct.internaltools.tools.templates.tool;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.internalads.model.AbstractTemplateResource;
import ru.yandex.direct.core.entity.internalads.model.DirectTemplate;
import ru.yandex.direct.core.entity.internalads.model.DirectTemplateResource;
import ru.yandex.direct.core.entity.internalads.model.DirectTemplateResourceOption;
import ru.yandex.direct.core.entity.internalads.model.DirectTemplateState;
import ru.yandex.direct.core.entity.internalads.model.InternalAdPlace;
import ru.yandex.direct.core.entity.internalads.model.TemplatePlace;
import ru.yandex.direct.core.entity.internalads.model.TemplateResource;
import ru.yandex.direct.core.entity.internalads.repository.DirectTemplatePlaceRepository;
import ru.yandex.direct.core.entity.internalads.repository.DirectTemplateRepository;
import ru.yandex.direct.core.entity.internalads.repository.DirectTemplateResourceRepository;
import ru.yandex.direct.core.entity.internalads.repository.PlaceRepository;
import ru.yandex.direct.core.entity.internalads.repository.TemplateResourceRepository;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolMassResult;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.templates.model.TemplateInfo;
import ru.yandex.direct.internaltools.tools.templates.model.TemplateInput;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CollectionDefects;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Strings.isNullOrEmpty;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static ru.yandex.direct.internaltools.tools.templates.model.TemplateInput.ACTION_LABEL;
import static ru.yandex.direct.internaltools.tools.templates.model.TemplateInput.DIRECT_TEMPLATE_ID_LABEL;
import static ru.yandex.direct.internaltools.tools.templates.model.TemplateInput.FORMAT_NAME_LABEL;
import static ru.yandex.direct.internaltools.tools.templates.model.TemplateInput.PLACE_IDS_LABEL;
import static ru.yandex.direct.internaltools.tools.templates.model.TemplateInput.RESOURCES_LABEL;
import static ru.yandex.direct.internaltools.tools.templates.model.TemplateInput.STATE_LABEL;
import static ru.yandex.direct.internaltools.tools.templates.tool.TemplateUtil.MAX_FORMAT_NAME_LENGTH;
import static ru.yandex.direct.internaltools.tools.templates.tool.TemplateUtil.MIN_NEW_TEMPLATE_ID;
import static ru.yandex.direct.internaltools.tools.templates.tool.TemplateUtil.SPECIFIC_VALUE_TO_CLEAR_PLACE_IDS;
import static ru.yandex.direct.internaltools.tools.templates.tool.TemplateUtil.UNIFIED_TEMPLATE_ID;
import static ru.yandex.direct.internaltools.tools.templates.tool.TemplateUtil.convertOptions;
import static ru.yandex.direct.internaltools.tools.templates.tool.TemplateUtil.getResourcesByNosMap;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.getLongIdsFromString;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.isStringWithValidLongIds;
import static ru.yandex.direct.internaltools.utils.ToolParameterUtils.parseLongLongLinkedMap;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.defect.NumberDefects.inInterval;

@Tool(
        consumes = TemplateInput.class,
        description = "Позволяет создавать, изменять, копировать и удалять шаблоны. Для операций копирования и " +
                "удаления учитывается ТОЛЬКО поле Id шаблона. Для операций создания и изменения под каждым полем " +
                "подписано, как оно используется.",
        label = "manage_direct_templates",
        name = "Управление шаблонами",
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@Category(InternalToolCategory.INTERNAL_ADS)
@ParametersAreNonnullByDefault
public class TemplateManagingTool implements BaseInternalTool<TemplateInput> {
    private static final Logger logger = LoggerFactory.getLogger(TemplateManagingTool.class);
    private static final Comparator<DirectTemplate> SORTING_COMPARATOR =
            Comparator.comparing(DirectTemplate::getDirectTemplateId).reversed();
    static final String TEMPLATE_STRING = "Template ";

    private final DirectTemplateRepository directTemplateRepository;
    private final DirectTemplatePlaceRepository directTemplatePlaceRepository;
    private final PlaceRepository placeRepository;
    private final DirectTemplateResourceRepository directTemplateResourceRepository;
    private final TemplateResourceRepository templateResourceRepository;

    @Autowired
    public TemplateManagingTool(DirectTemplateRepository directTemplateRepository,
                                DirectTemplatePlaceRepository directTemplatePlaceRepository,
                                PlaceRepository placeRepository,
                                DirectTemplateResourceRepository directTemplateResourceRepository,
                                TemplateResourceRepository templateResourceRepository) {
        this.directTemplateRepository = directTemplateRepository;
        this.directTemplatePlaceRepository = directTemplatePlaceRepository;
        this.placeRepository = placeRepository;
        this.directTemplateResourceRepository = directTemplateResourceRepository;
        this.templateResourceRepository = templateResourceRepository;
    }

    /**
     * Получаем информацию о шаблонах: id, имя формата, состояние перевода на единый формат,
     * список привязанных площадок и список ресурсов.
     */
    @Override
    public InternalToolResult processWithoutInput() {
        var directTemplateById = directTemplateRepository.getAll();
        var directTemplateIds = directTemplateById.keySet();
        var placesByTemplateId = directTemplatePlaceRepository.getByTemplateIds(directTemplateIds);
        var resourcesByTemplateId = directTemplateResourceRepository.getMapByTemplateIds(directTemplateIds);
        var unifiedTemplateResources = templateResourceRepository.getByTemplateIds(List.of(UNIFIED_TEMPLATE_ID));

        var templateInfoList = directTemplateById.values().stream()
                .sorted(SORTING_COMPARATOR)
                .map(directTemplate -> {
                    var directTemplateId = directTemplate.getDirectTemplateId();
                    return mapTemplateInfo(directTemplate, placesByTemplateId.get(directTemplateId),
                            resourcesByTemplateId.get(directTemplateId), unifiedTemplateResources);
                })
                .collect(Collectors.toList());

        return new InternalToolMassResult<>(templateInfoList);
    }

    private TemplateInfo mapTemplateInfo(DirectTemplate template,
                                         @Nullable List<TemplatePlace> templatePlaces,
                                         @Nullable List<DirectTemplateResource> templateResources,
                                         @Nullable List<TemplateResource> unifiedTemplateResources) {
        var resourcesStrings = mapResources(templateResources, unifiedTemplateResources);
        return new TemplateInfo()
                .withDirectTemplateId(template.getDirectTemplateId())
                .withFormatName(template.getFormatName())
                .withState(template.getState())
                .withPlaces(mapPlaces(templatePlaces))
                .withResourcesDetailed(resourcesStrings.getLeft())
                .withResourcesBasic(resourcesStrings.getRight());
    }

    private List<String> mapPlaces(@Nullable List<TemplatePlace> templatePlaces) {
        if (templatePlaces == null) {
            return List.of();
        }

        return mapList(templatePlaces, place -> place.getPlaceId().toString());
    }

    private Pair<List<String>, String> mapResources(@Nullable List<DirectTemplateResource> templateResources,
                                                    @Nullable List<TemplateResource> unifiedTemplateResources) {
        if (templateResources == null) {
            return Pair.of(List.of(), "");
        }

        var unifiedResourcesById = listToMap(unifiedTemplateResources, AbstractTemplateResource::getId);
        var resourcesBasic = new ArrayList<String>(templateResources.size());
        var resourcesDetailed = mapList(templateResources, resource -> {
            resourcesBasic.add(String.format("%d %d", resource.getUnifiedResourceNo(),
                    resource.getOptions().contains(DirectTemplateResourceOption.REQUIRED) ? 1 : 0));

            var resourceOptions = String.join(",", mapList(resource.getOptions(), Enum::toString));
            if (isNullOrEmpty(resourceOptions)) {
                resourceOptions = "empty options";
            }
            var unifiedResource = ifNotNull(unifiedResourcesById,
                    map -> map.get(resource.getUnifiedTemplateResourceId()));
            var resourceDescription = unifiedResource != null
                    ? unifiedResource.getDescription()
                    : "missing corresponding unified resource from ppcdict.template_resource";
            return String.format("[%d - %s - %s]", resource.getUnifiedResourceNo(), resourceOptions,
                    resourceDescription);
        });
        var resourcesBasicString = String.join(" | ", resourcesBasic);

        return Pair.of(resourcesDetailed, resourcesBasicString);
    }

    @Override
    @SuppressWarnings("rawtypes")
    public ValidationResult<TemplateInput, Defect> validate(TemplateInput input) {
        ItemValidationBuilder<TemplateInput, Defect> vb = ItemValidationBuilder.of(input);
        var action = input.getTemplateAction();
        vb.item(action, ACTION_LABEL)
                .check(notNull());
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }

        switch (action) {
            case UPDATE: {
                ItemValidationBuilder<Long, Defect> templateIdCheck = vb
                        .item(input.getDirectTemplateId(), DIRECT_TEMPLATE_ID_LABEL)
                        .check(notNull())
                        .checkBy(this::validateTemplateId, When.isValid());

                vb.item(input.getState(), STATE_LABEL)
                        .checkBy(state -> validateState(input.getDirectTemplateId(), state),
                                When.notNullAnd(When.isValidBoth(templateIdCheck)));

                vb.item(input.getFormatName(), FORMAT_NAME_LABEL)
                        .check(maxStringLength(MAX_FORMAT_NAME_LENGTH), When.isTrue(isNotBlank(input.getFormatName())))
                        .checkBy(formatName -> validateFormatName(input.getDirectTemplateId(), formatName),
                                When.notNullAnd(When.isValidBoth(templateIdCheck)));

                var isPlaceIdsEmpty = isBlank(input.getPlaceIds());
                vb.item(input.getPlaceIds(), PLACE_IDS_LABEL)
                        .check(isStringWithValidLongIds(1), When.isFalse(isPlaceIdsEmpty))
                        .checkBy(this::validatePlaceIds, When.isValidAnd(When.isFalse(isPlaceIdsEmpty)));

                vb.item(input.getResources(), RESOURCES_LABEL)
                        .checkBy(this::validateResources, When.isTrue(isNotBlank(input.getResources())));
                break;
            }
            case COPY:
            case DELETE:
                // требуется и учитывается только одно поле - directTemplateId
                vb.item(input.getDirectTemplateId(), DIRECT_TEMPLATE_ID_LABEL)
                        .check(notNull())
                        .checkBy(this::validateTemplateId, When.isValid());
                break;
            case CREATE: {
                vb.item(input.getFormatName(), FORMAT_NAME_LABEL)
                        .check(notNull())
                        .check(notBlank())
                        .check(maxStringLength(MAX_FORMAT_NAME_LENGTH), When.isValid())
                        .checkBy(formatName -> validateFormatName(null, formatName), When.isValid());

                var isPlaceIdsEmpty = isBlank(input.getPlaceIds());
                vb.item(input.getPlaceIds(), PLACE_IDS_LABEL)
                        .check(isStringWithValidLongIds(1), When.isFalse(isPlaceIdsEmpty))
                        .checkBy(this::validatePlaceIds, When.isValidAnd(When.isFalse(isPlaceIdsEmpty)));

                vb.item(input.getResources(), RESOURCES_LABEL)
                        .check(notNull())
                        .check(notBlank())
                        .checkBy(this::validateResources, When.isValid());
                break;
            }
            default:
                logger.warn("Action {} is not implemented yet", action);
        }

        return vb.getResult();
    }

    @SuppressWarnings("rawtypes")
    private ValidationResult<Long, Defect> validateTemplateId(Long id) {
        var vr = new ValidationResult<Long, Defect>(id);
        var existingTemplates = directTemplateRepository.get(List.of(id));
        if (!existingTemplates.containsKey(id)) {
            vr.addError(CommonDefects.objectNotFound());
        }
        return vr;
    }

    @SuppressWarnings("rawtypes")
    private ValidationResult<String, Defect> validateFormatName(@Nullable Long id, String formatName) {
        var vr = new ValidationResult<String, Defect>(formatName);
        // Только смигрированным (старым) шаблонам можно занулять имя формата
        if ((id == null || id >= MIN_NEW_TEMPLATE_ID)
                && formatName.trim().equals("0")) {
            logger.warn("У новых шаблонов не может отсутствовать имя формата");
            vr.addError(CommonDefects.invalidValue());
        }
        return vr;
    }

    @SuppressWarnings("rawtypes")
    private ValidationResult<DirectTemplateState, Defect> validateState(Long id, DirectTemplateState state) {
        var vr = new ValidationResult<DirectTemplateState, Defect>(state);
        if (id >= MIN_NEW_TEMPLATE_ID
                && state != DirectTemplateState.UNIFIED) {
            logger.warn("У новых/скопированных шаблонов может быть только статус UNIFIED");
            vr.addError(CommonDefects.invalidValue());
        }
        return vr;
    }

    @SuppressWarnings("rawtypes")
    private ValidationResult<String, Defect> validatePlaceIds(String placeIdsString) {
        var vr = new ValidationResult<String, Defect>(placeIdsString);
        Set<Long> placeIds = getLongIdsFromString(placeIdsString);
        var places = placeRepository.getByPlaceIds(placeIds);

        // Список площадок может содержать либо список id существующих площадок,
        // либо значение SPECIFIC_VALUE_TO_CLEAR_PLACE_IDS для отвязки всех площадок от шаблона
        var internalPlaceIds = mapList(places, InternalAdPlace::getId);
        if (!internalPlaceIds.containsAll(placeIds)
                && (placeIds.size() != 1 || !placeIds.contains(SPECIFIC_VALUE_TO_CLEAR_PLACE_IDS))) {
            vr.addError(CommonDefects.objectNotFound());
        }
        return vr;
    }

    @SuppressWarnings("rawtypes")
    private ValidationResult<String, Defect> validateResources(String resourcesString) {
        var vr = new ValidationResult<String, Defect>(resourcesString);
        Map<Long, Long> resourcesMap;
        try {
            resourcesMap = parseLongLongLinkedMap(resourcesString);
        } catch (NumberFormatException e) {
            logger.warn("Список ресурсов содержит нечисловые значения", e);
            vr.addError(CommonDefects.invalidValue());
            return vr;
        } catch (IllegalStateException e) {
            logger.warn("Список ресурсов содержит повторяющиеся номера ресурсов", e);
            vr.addError(CollectionDefects.duplicatedElement());
            return vr;
        } catch (ArrayIndexOutOfBoundsException e) {
            logger.warn("В парах ресурсов на какой-то строке нет минимум одного элемента", e);
            vr.addError(CommonDefects.invalidFormat());
            return vr;
        }

        // Должен быть как минимум 1 ресурс у шаблона
        if (resourcesMap.isEmpty()) {
            vr.addError(CollectionDefects.notEmptyCollection());
            return vr;
        }

        // Значения в словаре должны быть булевскими (0 или 1)
        var allowedValues = Set.of(0L, 1L);
        if (!allowedValues.containsAll(resourcesMap.values())) {
            logger.warn("В парах ресурсов есть значения, отличные от 0 или 1");
            vr.addError(inInterval(0, 1));
            return vr;
        }

        var expectedResourceNos = resourcesMap.keySet();
        var unifiedResourceNos = templateResourceRepository.getByTemplateIds(List.of(UNIFIED_TEMPLATE_ID)).stream()
                .map(AbstractTemplateResource::getResourceNo)
                .collect(Collectors.toSet());
        // все номера ресурсов должны существовать в template_resource и принадлежать единому шаблону
        if (!unifiedResourceNos.containsAll(expectedResourceNos)) {
            logger.warn("Среди номеров ресурсов есть не принадлежащие единому шаблону");
            vr.addError(CommonDefects.objectNotFound());
        }
        return vr;
    }

    @Override
    public InternalToolResult process(TemplateInput input) {
        var action = input.getTemplateAction();
        logger.info("{} template with params {}", action, input);

        switch (action) {
            case UPDATE:
                return update(input);
            case COPY:
                return copy(input);
            case CREATE:
                return create(input);
            case DELETE:
                return delete(input);
            default:
                return new InternalToolResult("Action " + action + " is not implemented yet");
        }
    }

    private InternalToolResult update(TemplateInput input) {
        var templateId = input.getDirectTemplateId();

        if (isNotBlank(input.getFormatName())) {
            directTemplateRepository.setFormatName(templateId, input.getFormatName().trim());
        }
        if (input.getState() != null) {
            directTemplateRepository.setState(templateId, input.getState());
        }

        if (isNotBlank(input.getPlaceIds())) {
            Set<Long> newPlaceIds = getLongIdsFromString(input.getPlaceIds());

            var savedPlaces = directTemplatePlaceRepository.getByTemplateId(templateId);
            var savedPlaceIds = new HashSet<>(mapList(savedPlaces, TemplatePlace::getPlaceId));

            // если newPlaceIds содержит только SPECIFIC_VALUE_TO_CLEAR_PLACE_IDS, то удалятся все записи
            var placesToDelete = savedPlaces.stream()
                    .filter(p -> !newPlaceIds.contains(p.getPlaceId()))
                    .collect(Collectors.toList());
            directTemplatePlaceRepository.delete(placesToDelete);

            var placeIdsToAdd = new HashSet<>(newPlaceIds);
            placeIdsToAdd.removeAll(savedPlaceIds);
            placeIdsToAdd.remove(SPECIFIC_VALUE_TO_CLEAR_PLACE_IDS);
            var placesToAdd = mapList(placeIdsToAdd, id -> new TemplatePlace()
                    .withTemplateId(templateId)
                    .withPlaceId(id));
            directTemplatePlaceRepository.add(placesToAdd);
        }

        // Обновляем ресурсы
        if (isNotBlank(input.getResources())) {
            var requiredOptionByUnifiedResourceNo = parseLongLongLinkedMap(input.getResources());
            var expectedUnifiedResourceNos = requiredOptionByUnifiedResourceNo.keySet();

            var existingResources = directTemplateResourceRepository.getByTemplateIds(List.of(templateId));
            var existingResourceIdByUnifiedResourceNo = listToMap(existingResources,
                    DirectTemplateResource::getUnifiedResourceNo, DirectTemplateResource::getDirectTemplateResourceId);

            var unifiedResourceNosToDelete = new HashSet<>(existingResourceIdByUnifiedResourceNo.keySet());
            unifiedResourceNosToDelete.removeAll(expectedUnifiedResourceNos);

            var unifiedResourceNosToAdd = new HashSet<>(expectedUnifiedResourceNos);
            unifiedResourceNosToAdd.removeAll(existingResourceIdByUnifiedResourceNo.keySet());

            // удаление убранных ресурсов
            var resourceIdsToDelete = mapList(unifiedResourceNosToDelete, existingResourceIdByUnifiedResourceNo::get);
            directTemplateResourceRepository.delete(resourceIdsToDelete);

            // обновление опции REQUIRED для измененных ресурсов
            var resourcesToUpdate = existingResources.stream()
                    .filter(resource -> {
                        // пропускаем уже удаленные ресурсы
                        if (resourceIdsToDelete.contains(resource.getDirectTemplateResourceId())) {
                            return false;
                        }
                        var unifiedResourceNo = resource.getUnifiedResourceNo();
                        var existingRequired = resource.getOptions().contains(DirectTemplateResourceOption.REQUIRED);
                        var expectedRequired = requiredOptionByUnifiedResourceNo.get(unifiedResourceNo) == 1L;
                        return existingRequired != expectedRequired;
                    })
                    .map(resource -> {
                        var options = new HashSet<>(resource.getOptions());
                        if (requiredOptionByUnifiedResourceNo.get(resource.getUnifiedResourceNo()) == 1L) {
                            options.add(DirectTemplateResourceOption.REQUIRED);
                        } else {
                            options.remove(DirectTemplateResourceOption.REQUIRED);
                        }
                        resource.setOptions(options);
                        return resource;
                    })
                    .collect(Collectors.toList());
            directTemplateResourceRepository.addOrUpdate(resourcesToUpdate);

            // добавление новых ресурсов
            var requiredOptionByUnifiedResourceNoToAdd = EntryStream.of(requiredOptionByUnifiedResourceNo)
                    .filterKeys(unifiedResourceNosToAdd::contains)
                    .toCustomMap(LinkedHashMap::new);
            if (!requiredOptionByUnifiedResourceNoToAdd.isEmpty()) {
                Long maxResourceNo = null;
                // для смигрированных шаблонов находим максимальный номер ресурса
                if (templateId < MIN_NEW_TEMPLATE_ID) {
                    // существовавший ресурс должен быть как минимум один, даже если мы его уже удалили выше,
                    // но перестрахуемся на случай ошибки в данных
                    var resourceWithMaxResourceNo = existingResources.stream()
                            .max(Comparator.comparingLong(DirectTemplateResource::getResourceNo));
                    maxResourceNo = resourceWithMaxResourceNo
                            .map(DirectTemplateResource::getResourceNo)
                            .orElse(null);
                }
                createResources(templateId, requiredOptionByUnifiedResourceNoToAdd, maxResourceNo);
            }
        }

        return new InternalToolResult(TEMPLATE_STRING + templateId + " has been successfully updated");
    }

    private InternalToolResult copy(TemplateInput input) {
        var templateId = input.getDirectTemplateId();

        // Копируем данные в direct_template. State у скопированного шаблона должен быть UNIFIED
        var existingTemplate = directTemplateRepository.get(List.of(templateId)).get(templateId);
        var newTemplate = new DirectTemplate()
                .withFormatName(existingTemplate.getFormatName())
                .withState(DirectTemplateState.UNIFIED);
        var newTemplateId = directTemplateRepository.add(newTemplate);

        // Копируем данные в direct_template_place
        var existingPlaces = directTemplatePlaceRepository.getByTemplateId(templateId);
        var newPlaces = mapList(existingPlaces, place -> place.withTemplateId(newTemplateId));
        directTemplatePlaceRepository.add(newPlaces);

        // Копируем данные в direct_template_resource
        var existingResources = directTemplateResourceRepository.getByTemplateIds(List.of(templateId));
        var newResources = mapList(existingResources, resource -> resource
                .withDirectTemplateId(newTemplateId)
                .withDirectTemplateResourceId(null));
        directTemplateResourceRepository.addOrUpdate(newResources);

        return new InternalToolResult(TEMPLATE_STRING + newTemplateId + " has been successfully copied from template "
                + templateId);
    }

    private InternalToolResult create(TemplateInput input) {
        // Создаем запись в direct_template. State у созданного шаблона должен быть UNIFIED
        var newTemplate = new DirectTemplate()
                .withFormatName(input.getFormatName().trim())
                .withState(DirectTemplateState.UNIFIED);
        var newTemplateId = directTemplateRepository.add(newTemplate);

        // Создаем записи в direct_template_place, если во входных данных есть площадки
        Set<Long> newPlaceIds = isNotBlank(input.getPlaceIds())
                ? new HashSet<>(getLongIdsFromString(input.getPlaceIds()))
                : new HashSet<>();
        newPlaceIds.remove(SPECIFIC_VALUE_TO_CLEAR_PLACE_IDS);
        var newPlaces = mapList(newPlaceIds, placeId -> new TemplatePlace()
                .withTemplateId(newTemplateId)
                .withPlaceId(placeId));
        directTemplatePlaceRepository.add(newPlaces);

        // Создаем записи в direct_template_resource
        var requiredOptionByUnifiedResourceNo = parseLongLongLinkedMap(input.getResources());
        createResources(newTemplateId, requiredOptionByUnifiedResourceNo, null);

        return new InternalToolResult(TEMPLATE_STRING + newTemplateId + " has been successfully created");
    }

    private InternalToolResult delete(TemplateInput input) {
        var templateId = input.getDirectTemplateId();

        var existingTemplate = directTemplateRepository.get(List.of(templateId)).get(templateId);
        logger.info("Deleted template data: {}", existingTemplate);
        directTemplateRepository.delete(templateId);

        var existingPlaces = directTemplatePlaceRepository.getByTemplateId(templateId);
        logger.info("Deleted template places: {}", existingPlaces);
        directTemplatePlaceRepository.delete(existingPlaces);

        var existingResources = directTemplateResourceRepository.getByTemplateIds(List.of(templateId));
        var directTemplateResourceIds = mapList(existingResources, DirectTemplateResource::getDirectTemplateResourceId);
        logger.info("Deleted template resources: {}", existingResources);
        directTemplateResourceRepository.delete(directTemplateResourceIds);

        return new InternalToolResult(TEMPLATE_STRING + templateId + " has been successfully deleted");
    }

    /**
     * Создает ресурсы для шаблона в том порядке, в котором они переданы в метод.
     * <p>
     * Если значение maxResourceNo задано, то номера ресурсов создаваемых ресурсов будут выше этого значения;
     * если значение maxResourceNo = null, то номера ресурсов будут совпадать с номерами ресурсов единого шаблона.
     *
     * @param templateId                        id шаблона
     * @param requiredOptionByUnifiedResourceNo словарь
     * @param maxResourceNo                     значение, выше которого должны быть создаваемые номера ресурсов
     */
    private void createResources(Long templateId, Map<Long, Long> requiredOptionByUnifiedResourceNo,
                                 @Nullable Long maxResourceNo) {
        var unifiedResources = templateResourceRepository.getByTemplateIds(List.of(UNIFIED_TEMPLATE_ID));
        var unifiedResourcesByNos = getResourcesByNosMap(requiredOptionByUnifiedResourceNo.keySet(), unifiedResources,
                UNIFIED_TEMPLATE_ID);

        var newResources = new ArrayList<DirectTemplateResource>();
        for (var entry : requiredOptionByUnifiedResourceNo.entrySet()) {
            var unifiedResourceNo = entry.getKey();
            var resourceNo = maxResourceNo == null
                    ? unifiedResourceNo
                    : maxResourceNo + unifiedResourceNo;
            var unifiedResource = unifiedResourcesByNos.get(unifiedResourceNo);

            // Берем options из ресурса единого шаблона и подменяем опцию required на переданную в инструменте
            boolean isRequired = entry.getValue() == 1L;
            var unifiedResourceOptions = convertOptions(unifiedResource.getOptions());
            if (isRequired) {
                unifiedResourceOptions.add(DirectTemplateResourceOption.REQUIRED);
            } else {
                unifiedResourceOptions.remove(DirectTemplateResourceOption.REQUIRED);
            }

            newResources.add(new DirectTemplateResource()
                    .withDirectTemplateId(templateId)
                    .withResourceNo(resourceNo)
                    .withUnifiedResourceNo(unifiedResourceNo)
                    .withUnifiedTemplateResourceId(unifiedResource.getId())
                    .withOptions(unifiedResourceOptions));
        }
        directTemplateResourceRepository.addOrUpdate(newResources);
    }
}
