package ru.yandex.direct.internaltools.tools.templates.tool;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.internalads.model.DirectTemplate;
import ru.yandex.direct.core.entity.internalads.model.DirectTemplateResource;
import ru.yandex.direct.core.entity.internalads.model.DirectTemplateState;
import ru.yandex.direct.core.entity.internalads.model.TemplateResource;
import ru.yandex.direct.core.entity.internalads.repository.DirectTemplatePlaceRepository;
import ru.yandex.direct.core.entity.internalads.repository.DirectTemplateRepository;
import ru.yandex.direct.core.entity.internalads.repository.DirectTemplateResourceRepository;
import ru.yandex.direct.core.entity.internalads.repository.TemplatePlaceRepository;
import ru.yandex.direct.core.entity.internalads.repository.TemplateResourceRepository;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.templates.model.TemplateResourceMapping;
import ru.yandex.direct.internaltools.utils.ToolParameterUtils;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.constraint.StringConstraints;
import ru.yandex.direct.validation.defect.CollectionDefects;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.internaltools.tools.templates.model.TemplateResourceMapping.FORMAT_LABEL;
import static ru.yandex.direct.internaltools.tools.templates.model.TemplateResourceMapping.RESOURCE_NO_MAPPING_LABEL;
import static ru.yandex.direct.internaltools.tools.templates.model.TemplateResourceMapping.TEMPLATE_ID_LABEL;
import static ru.yandex.direct.internaltools.tools.templates.tool.TemplateUtil.UNIFIED_TEMPLATE_ID;
import static ru.yandex.direct.internaltools.tools.templates.tool.TemplateUtil.convertOptions;
import static ru.yandex.direct.internaltools.tools.templates.tool.TemplateUtil.getResourcesByNosMap;

@Tool(
        consumes = TemplateResourceMapping.class,
        description = "Мигрирует старый шаблон на единый формат"
                + " через задание маппинга старых ресурсов в ресурсы единого шаблона",
        label = "map_template_resources",
        name = "Миграция шаблона на единый формат",
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@Category(InternalToolCategory.INTERNAL_ADS)
@ParametersAreNonnullByDefault
public class TemplateMappingTool implements BaseInternalTool<TemplateResourceMapping> {
    private static final Logger LOGGER = LoggerFactory.getLogger(TemplateMappingTool.class);

    private final DirectTemplateRepository directTemplateRepository;
    private final DirectTemplateResourceRepository directTemplateResourceRepository;
    private final DirectTemplatePlaceRepository directTemplatePlaceRepository;
    private final TemplateResourceRepository templateResourceRepository;
    private final TemplatePlaceRepository templatePlaceRepository;

    @Autowired
    public TemplateMappingTool(DirectTemplateRepository directTemplateRepository,
            DirectTemplateResourceRepository directTemplateResourceRepository,
            DirectTemplatePlaceRepository directTemplatePlaceRepository,
            TemplateResourceRepository templateResourceRepository,
            TemplatePlaceRepository templatePlaceRepository) {
        this.directTemplateRepository = directTemplateRepository;
        this.directTemplateResourceRepository = directTemplateResourceRepository;
        this.directTemplatePlaceRepository = directTemplatePlaceRepository;
        this.templateResourceRepository = templateResourceRepository;
        this.templatePlaceRepository = templatePlaceRepository;
    }

    @Override
    @SuppressWarnings("rawtypes")
    public ValidationResult<TemplateResourceMapping, Defect> validate(
            TemplateResourceMapping param) {
        ItemValidationBuilder<TemplateResourceMapping, Defect> vb = ItemValidationBuilder.of(param);
        vb.item(param.getTemplateId(), TEMPLATE_ID_LABEL)
                .check(CommonConstraints.notNull())
                .checkBy(id -> validateTemplateId(id, param.isRedoMigration()));
        vb.item(param.getFormat(), FORMAT_LABEL)
                .check(StringConstraints.notBlank());
        vb.item(param.getMapping(), RESOURCE_NO_MAPPING_LABEL)
                .check(StringConstraints.notBlank())
                .checkBy(mapping -> validateMapping(mapping, param.getTemplateId()), When.isValid());
        return vb.getResult();
    }

    @SuppressWarnings("rawtypes")
    private ValidationResult<String, Defect> validateMapping(String mapping, Long templateId) {
        var vr = new ValidationResult<String, Defect>(mapping);
        Map<Long, Long> map;
        try {
            map = ToolParameterUtils.parseLongLongMap(mapping);
        } catch (IllegalStateException e) {
            LOGGER.error("В маппинге есть повторяющиеся ключи", e);
            vr.addError(CollectionDefects.duplicatedElement());
            return vr;
        } catch (ArrayIndexOutOfBoundsException e) {
            LOGGER.error("В маппинге на какой-то строке нет второго элемента", e);
            vr.addError(CommonDefects.invalidFormat());
            return vr;
        }

        if (map.isEmpty()) { // не должно случаться
            vr.addError(CollectionDefects.notEmptyCollection()); // по ощущениям перепутаны названия дефектов
            return vr;
        }

        if (map.values().stream().distinct().count() != map.size()) {
            LOGGER.error("В маппинге есть повторяющиеся значения");
            vr.addError(CollectionDefects.duplicatedElement());
            return vr;
        }

        var resources = templateResourceRepository.getByTemplateIds(List.of(templateId, UNIFIED_TEMPLATE_ID));
        var oldResourcesByNos = getResourcesByNosMap(map.keySet(), resources, templateId);
        var unifiedResourcesByNos = getResourcesByNosMap(map.values(), resources, UNIFIED_TEMPLATE_ID);
        if (checkResourcesExistence(map.keySet(), oldResourcesByNos)
                || checkResourcesExistence(map.values(), unifiedResourcesByNos)) {
            LOGGER.error("Не найден один из указанных ресурсов");
            vr.addError(CommonDefects.objectNotFound());
            return vr;
        }
        if (checkResourcesOptions(map, oldResourcesByNos, unifiedResourcesByNos)) {
            LOGGER.error("Не совпадают опции banana_url или banana_image у ресурсов");
            vr.addError(CommonDefects.inconsistentState());
        }
        return vr;
    }

    // возвращает true, если у ресурсов не совпадают опции banana_url или banana_image
    private boolean checkResourcesOptions(Map<Long, Long> map,
            Map<Long, TemplateResource> oldResourcesByNos,
            Map<Long, TemplateResource> unifiedResourcesByNos) {
        for (var entry : map.entrySet()) {
            if ((oldResourcesByNos.get(entry.getKey()).isBananaImage()
                    != unifiedResourcesByNos.get(entry.getValue()).isBananaImage())
                || (oldResourcesByNos.get(entry.getKey()).isBananaUrl()
                    != unifiedResourcesByNos.get(entry.getValue()).isBananaUrl())) {
                return true;
            }
        }
        return false;
    }

    // вернуть true, если какой-то ресурс старого или единого шаблона не найден
    private boolean checkResourcesExistence(Collection<Long> resourceNos,
            Map<Long, TemplateResource> resourcesByNos) {
        return !resourcesByNos.keySet().containsAll(resourceNos);
    }

    @SuppressWarnings("rawtypes")
    private ValidationResult<Long, Defect> validateTemplateId(Long id, boolean redoMigration) {
        var vr = new ValidationResult<Long, Defect>(id);
        var dbResources = templateResourceRepository.getByTemplateIds(List.of(id));
        if (dbResources.isEmpty()) {
            vr.addError(CommonDefects.objectNotFound());
        }
        var dbTemplate = directTemplateRepository.get(List.of(id)).get(id);
        if (dbTemplate != null && !redoMigration) {
            vr.addError(CommonDefects.inconsistentStateAlreadyExists());
        }
        return vr;
    }

    private void clearMapping(long templateId) {
        var templateData = directTemplateRepository.get(List.of(templateId)).get(templateId);
        if (templateData != null) {
            LOGGER.info("Old direct template data: {}", templateData);
            directTemplateRepository.delete(templateId);
        }

        var templatePlaceData = directTemplatePlaceRepository.getByTemplateId(templateId);
        if (templatePlaceData != null) {
            LOGGER.info("Old direct template place data: {}", templatePlaceData);
            directTemplatePlaceRepository.delete(templatePlaceData);
        }

        var templateResourceData = directTemplateResourceRepository.getByTemplateIds(List.of(templateId));
        if (templateResourceData != null) {
            LOGGER.info("Old direct template resource data: {}", templateResourceData);
            directTemplateResourceRepository.delete(
                    templateResourceData.stream()
                            .map(DirectTemplateResource::getDirectTemplateResourceId)
                            .collect(Collectors.toList())
            );
        }
    }

    @Override
    public InternalToolResult process(TemplateResourceMapping param) {
        var templateId = param.getTemplateId();
        LOGGER.info("Migrating template {} with params {}", templateId, param);

        if (param.isRedoMigration()) {
            clearMapping(templateId);
        }

        var directTemplate = new DirectTemplate()
                .withDirectTemplateId(templateId)
                .withFormatName(param.getFormat().trim())
                .withState(DirectTemplateState.DEFAULT);

        directTemplateRepository.addOldTemplate(directTemplate);
        LOGGER.info("Added template data");

        var templatePlaces = templatePlaceRepository.getByTemplateId(templateId);
        directTemplatePlaceRepository.add(templatePlaces);

        var map = ToolParameterUtils.parseLongLongMap(param.getMapping());
        var resources = templateResourceRepository.getByTemplateIds(List.of(templateId, UNIFIED_TEMPLATE_ID));
        var oldResourcesByNos = getResourcesByNosMap(map.keySet(), resources, templateId);
        var unifiedResourcesByNos = getResourcesByNosMap(map.values(), resources, UNIFIED_TEMPLATE_ID);

        var newResources = new ArrayList<DirectTemplateResource>();
        for (var entry : map.entrySet()) {
            newResources.add(new DirectTemplateResource()
                    .withDirectTemplateResourceId(oldResourcesByNos.get(entry.getKey()).getId())
                    .withDirectTemplateId(templateId)
                    .withResourceNo(entry.getKey())
                    .withUnifiedResourceNo(entry.getValue())
                    .withUnifiedTemplateResourceId(unifiedResourcesByNos.get(entry.getValue()).getId())
                    .withOptions(convertOptions(oldResourcesByNos.get(entry.getKey()).getOptions()))
            );
        }
        directTemplateResourceRepository.addOrUpdate(newResources);
        return new InternalToolResult("OK");
    }
}
