package ru.yandex.direct.internaltools.tools.traceinterceptions;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolMassResult;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.traceinterceptions.model.TraceInterceptionConverter;
import ru.yandex.direct.internaltools.tools.traceinterceptions.model.TraceInterceptionInfo;
import ru.yandex.direct.internaltools.tools.traceinterceptions.model.TraceInterceptionUpdateActionInput;
import ru.yandex.direct.traceinterception.entity.traceinterception.service.TraceInterceptionsManagingService;
import ru.yandex.direct.traceinterception.model.TraceInterception;
import ru.yandex.direct.traceinterception.model.TraceInterceptionAction;

import static ru.yandex.direct.internaltools.tools.traceinterceptions.model.TraceInterceptionUpdateSelectProvider.extractId;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Tool(
        name = "Редактирование действий правил перехвата",
        label = "edit_trace_interceptions_action",
        description = "Редактирование правил перехвата в Директе. Можно добавить, поменять или отредактировать действие." +
                "Переключение статуса правила на другой странице.",
        consumes = TraceInterceptionUpdateActionInput.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.TRACE_INTERCEPTIONS)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class TraceInterceptionsUpdateActionTool implements BaseInternalTool<TraceInterceptionUpdateActionInput> {
    TraceInterceptionsManagingService managingService;

    public TraceInterceptionsUpdateActionTool(TraceInterceptionsManagingService traceInterceptionsManagingService) {
        this.managingService = traceInterceptionsManagingService;
    }

    @Override
    public InternalToolMassResult<TraceInterceptionInfo> processWithoutInput() {
        return new InternalToolMassResult<>(mapList(managingService.getCached(),
                TraceInterceptionConverter::createTraceInterceptionInfo));
    }

    @Override
    public final InternalToolResult process(TraceInterceptionUpdateActionInput input) {
        Long inputId = extractId(input.getId());
        TraceInterception traceInterception = managingService.getCached().stream()
                .filter(x -> inputId.equals(x.getId()))
                .findFirst()
                .orElseThrow(() -> new IllegalArgumentException(("Can't find interception with id=" + input.getId())));

        TraceInterceptionAction action = traceInterception.getAction();
        String actionValue = input.getActionValue();
        switch (input.getActionType()) {
            case SLEEP:
                action.setSleepDuration(actionValue != null ? Long.parseLong(actionValue) : null);
                break;
            case EXCEPTION:
                action.setExceptionMessage(actionValue);
                break;
            case SEMAPHORE:
                action.setSemaphorePermits(actionValue != null ? Integer.parseInt(actionValue) : null);
                break;
            case SEMAPHORE_KEY:
                action.setSemaphoreKey(actionValue);
                break;
            default:
                throw new IllegalArgumentException("Unknown type");
        }
        managingService.updateInterceptionWithActionById(traceInterception.getId(), action);
        return processWithoutInput();
    }
}
