package ru.yandex.direct.internaltools.tools.trackingphone;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.spring.TestingComponent;
import ru.yandex.direct.core.entity.campcalltrackingphones.repository.CampCalltrackingPhonesRepository;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneService;
import ru.yandex.direct.core.entity.clientphone.repository.ClientPhoneRepository;
import ru.yandex.direct.core.entity.trackingphone.model.ClientPhone;
import ru.yandex.direct.core.entity.trackingphone.model.PhoneNumber;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.trackingphone.model.TelephonyPhoneInfo;
import ru.yandex.direct.internaltools.tools.trackingphone.model.UnlinkTelephonyToolParameter;

import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Tool(
        name = "Отвязка номеров телефонии",
        label = "unlink_telephony",
        description = "Показывает, какие номера используются, и дает отвзять номер. " +
                "Если номер использовался на баннере, то он заменяется на номер организации.",
        consumes = UnlinkTelephonyToolParameter.class,
        type = InternalToolType.WRITER
)
@TestingComponent
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.CALLTRACKING)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class UnlinkTelephonyTool extends MassInternalTool<UnlinkTelephonyToolParameter, TelephonyPhoneInfo> {

    @Autowired
    private ClientPhoneRepository clientPhoneRepository;

    @Autowired
    private CampCalltrackingPhonesRepository campCalltrackingPhonesRepository;

    @Autowired
    private ShardHelper shardHelper;

    @Autowired
    private ClientService clientService;

    @Autowired
    private TelephonyPhoneService telephonyPhoneService;

    @Override
    protected List<TelephonyPhoneInfo> getMassData(UnlinkTelephonyToolParameter parameter) {

        if (UnlinkTelephonyToolParameter.Action.UNLINK_PHONE.equals(parameter.getAction())) {
            unlinkNumber(parameter.getServiceId());
        }

        return getAllTelephonyPhones();
    }

    @Override
    protected List<TelephonyPhoneInfo> getMassData() {
        return getAllTelephonyPhones();
    }

    private void unlinkNumber(String serviceId) {
        telephonyPhoneService.detachTelephony(serviceId);
        AtomicReference<ClientPhone> phoneRef = new AtomicReference<>();
        AtomicInteger phoneShardRef = new AtomicInteger();
        shardHelper.forEachShard(shard -> {
            List<ClientPhone> phones = clientPhoneRepository.getPhonesByServiceId(shard, serviceId);
            if (!phones.isEmpty()) {
                phoneShardRef.set(shard);
                phoneRef.set(phones.get(0));
            }
        });

        ClientPhone phone = phoneRef.get();
        int phoneShard = phoneShardRef.get();

        List<Long> phoneIds = List.of(phone.getId());

        clientPhoneRepository.delete(phone.getClientId(), phoneIds);

        if (phone.getPermalinkId() == null) {
            campCalltrackingPhonesRepository.delete(phoneShard, phoneIds);
        } else {
            clientPhoneRepository.unlinkBannerPhonesByPhoneId(phoneShard, phoneIds);
            clientPhoneRepository.unlinkCampaignPhonesByPhoneId(phoneShard, phoneIds);
        }
    }

    private List<TelephonyPhoneInfo> getAllTelephonyPhones() {
        List<ClientPhone> phones = new ArrayList<>();

        shardHelper.forEachShard(shard -> phones.addAll(clientPhoneRepository.getAllTelephonyPhones(shard)));

        List<Long> clientIds = StreamEx.of(phones)
                .map(ClientPhone::getClientId)
                .map(ClientId::asLong)
                .distinct()
                .toList();

        Map<Long, String> clientIdToLogin = StreamEx.of(clientService.massGetClientWithUsers(clientIds))
                .mapToEntry(Client::getClientId, Function.identity())
                .mapValues(clientWithUsers -> {
                    List<User> users = clientWithUsers.getUsers();
                    return filterList(users, user -> user.getUid().equals(clientWithUsers.getChiefUid())).get(0).getLogin();
                })
                .toMap();

        return mapList(phones, phone -> convertToTelephonyPhoneInfo(clientIdToLogin, phone));
    }

    private TelephonyPhoneInfo convertToTelephonyPhoneInfo(Map<Long, String> clientIdToLogin, ClientPhone phone) {
        PhoneNumber phoneNumber = phone.getTelephonyPhone();
        String redirectPhone = phone.getPhoneNumber().getPhone();
        return new TelephonyPhoneInfo()
                .withLogin(clientIdToLogin.get(phone.getClientId().asLong()))
                .withClientPhoneId(phone.getId())
                .withCounterId(phone.getCounterId())
                .withPermalinkId(phone.getPermalinkId())
                .withPhone(phoneNumber.getPhone())
                .withRedirectPhone(redirectPhone)
                .withTelephonyServiceId(phone.getTelephonyServiceId())
                .withIsDeleted(phone.getIsDeleted());
    }
}
