package ru.yandex.direct.internaltools.tools.turbo;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.banner.model.TurboAppInfo;
import ru.yandex.direct.core.entity.banner.type.turboapp.TurboAppsInfoRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolException;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.turbo.model.IntToolTurboApp;
import ru.yandex.direct.internaltools.tools.turbo.model.TurboAppAddParameters;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.dbutil.sharding.ShardSupport.NO_SHARD;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Tool(
        name = "Добавление турбо-аппа для клиента",
        label = "add_client_turbo_app",
        description = "Добавить турбо-апп клиента. \n" +
                "Чтобы узнать текущий турбо-апп клиента, оставьте поле идентификатора турбо-аппа пустым.",
        consumes = TurboAppAddParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.ADD)
@Category(InternalToolCategory.TURBO_APP)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@ParametersAreNonnullByDefault
public class TurboAppAddToClientTool extends MassInternalTool<TurboAppAddParameters, IntToolTurboApp> {

    private final ShardHelper shardHelper;
    private final TurboAppsInfoRepository turboAppsInfoRepository;

    public TurboAppAddToClientTool(
            ShardHelper shardHelper,
            TurboAppsInfoRepository turboAppsInfoRepository
    ) {
        this.shardHelper = shardHelper;
        this.turboAppsInfoRepository = turboAppsInfoRepository;
    }

    @Override
    public ValidationResult<TurboAppAddParameters, Defect> validate(TurboAppAddParameters params) {
        ItemValidationBuilder<TurboAppAddParameters, Defect> vb = ItemValidationBuilder.of(params, Defect.class);
        vb.item(params.getClientId(), "clientId")
                .check(notNull())
                .check(validId());
        vb.item(params.getTurboAppId(), "turboAppId")
                .check(validId());
        return vb.getResult();
    }

    @Override
    protected List<IntToolTurboApp> getMassData(TurboAppAddParameters parameters) {
        Long turboAppId = parameters.getTurboAppId();
        Long pClientId = parameters.getClientId();
        ClientId clientId = ClientId.fromLong(pClientId);
        int shard = shardHelper.getShardByClientId(clientId);
        if (shard == NO_SHARD) {
            throw new InternalToolException("Nonexistent client: " + clientId);
        }
        List<TurboAppInfo> turboAppInfos = turboAppsInfoRepository.getTurboAppInfoByClientId(shard, clientId);
        if (turboAppId == null) {
            if (isEmpty(turboAppInfos)) {
                throw new InternalToolException("Client " + clientId + " doesn't have any turbo apps");
            }
        } else {
            Set<Long> clientTurboAppIds = listToSet(turboAppInfos, TurboAppInfo::getTurboAppId);
            if (!clientTurboAppIds.contains(turboAppId)) {
                var turboAppInfo = new TurboAppInfo()
                        .withClientId(pClientId)
                        .withTurboAppId(turboAppId)
                        .withContent(parameters.getMetaContent());
                turboAppsInfoRepository.addOrUpdateTurboAppInfo(shard, List.of(turboAppInfo));
                turboAppInfos = turboAppsInfoRepository.getTurboAppInfoByClientId(shard, clientId);
            }
        }
        return mapList(turboAppInfos, t -> convertToIntTool(pClientId, t.getTurboAppId(), t.getContent()));
    }

    private IntToolTurboApp convertToIntTool(Long clientId, Long turboAppId, String metaContent) {
        return new IntToolTurboApp()
                .withClientId(clientId)
                .withTurboAppId(turboAppId)
                .withMetaContent(metaContent);
    }
}
