package ru.yandex.direct.internaltools.tools.uac

import org.springframework.beans.factory.annotation.Qualifier
import ru.yandex.direct.core.entity.client.model.Client
import ru.yandex.direct.core.entity.client.service.ClientService
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toAdvType
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toEAdCampaignType
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.service.UacBannerService
import ru.yandex.direct.core.grut.api.ClientGrutModel
import ru.yandex.direct.core.grut.api.DefaultGrutApiProperties
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.internaltools.configuration.InternalToolsConfiguration.GRUT_CONTEXT_FOR_PROD
import ru.yandex.direct.internaltools.core.BaseInternalTool
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup
import ru.yandex.direct.internaltools.core.annotations.tool.Action
import ru.yandex.direct.internaltools.core.annotations.tool.Category
import ru.yandex.direct.internaltools.core.annotations.tool.Disclaimers
import ru.yandex.direct.internaltools.core.annotations.tool.HideInProduction
import ru.yandex.direct.internaltools.core.annotations.tool.Tool
import ru.yandex.direct.internaltools.core.container.InternalToolResult
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole
import ru.yandex.direct.internaltools.core.enums.InternalToolAction
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory
import ru.yandex.direct.internaltools.core.enums.InternalToolType
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException
import ru.yandex.direct.internaltools.tools.uac.model.CampaignTeleporterParam
import ru.yandex.direct.internaltools.tools.uac.tool.toAssetGrut
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.constraint.CommonConstraints.validId
import ru.yandex.direct.validation.util.property
import ru.yandex.direct.validation.util.validateObject
import ru.yandex.grut.objects.proto.client.Schema

private const val DESCRIPTION =
    """Копирование GRuT заявки на DT/TS. Что делает инструмент:"""

@Tool(
    name = "Телепортер GRuT заявок",
    label = "campaign_teleporter_tool",
    description = DESCRIPTION,
    consumes = CampaignTeleporterParam::class,
    type = InternalToolType.WRITER
)
@Action(InternalToolAction.EXECUTE)
@Category(InternalToolCategory.UAC)
@AccessGroup(InternalToolAccessRole.INTERNAL_USER)
@HideInProduction
@Disclaimers(
    "Проверяет на DT/TS наличие клиента в груте и создает нового при его отсутствии",
    "Копирует ассеты из прода на DT/TS, с удалением существующих ассетов кампании на DT/TS (если она существует)",
    "Копирует заявку из прода на DT/TS",
    "Ставит заявку в очередь UpdateAdsJob на перегенерацию объявлений (при выставлении соотв. галки)",
)
class CampaignTeleporterTool(
    private val shardHelper: ShardHelper,
    private val clientService: ClientService,
    private val uacBannerService: UacBannerService,
    private val grutTransactionProvider: GrutTransactionProvider,
    private val grutApiService: GrutApiService,
    @Qualifier(GRUT_CONTEXT_FOR_PROD) private val prodGrutContext: GrutContext,
) : BaseInternalTool<CampaignTeleporterParam> {
    private val prodGrutApiService = GrutApiService(prodGrutContext, DefaultGrutApiProperties())

    override fun validate(param: CampaignTeleporterParam) = validateObject(param) {
        property(param::campaignId)
            .check(notNull())
            .check(validId(), When.isValid())
    }

    override fun process(param: CampaignTeleporterParam): InternalToolResult {
        val campaignId = param.campaignId
        val prodTCampaign = prodGrutApiService.briefGrutApi.getBrief(campaignId)
        val targetTCampaign = grutApiService.briefGrutApi.getBrief(campaignId)
        if (prodTCampaign == null) {
            throw InternalToolValidationException("Нет кампании $campaignId в продакшене GRuT")
        }

        if (targetTCampaign != null) {
            if (!param.updateIfExist) {
                throw InternalToolValidationException("Кампания $campaignId уже есть на DT/TS в GRuT. " +
                    "Поставьте галочку на соотв. пункте выше, чтобы обновить существующую заявку.")
            } else if (targetTCampaign.meta.campaignType.toAdvType() != prodTCampaign.meta.campaignType.toAdvType()) {
                throw InternalToolValidationException("Разные типы кампаний у заявок в продакшене " +
                    "'${prodTCampaign.meta.campaignType.toAdvType()}' " +
                    "и на DT/TS '${targetTCampaign.meta.campaignType.toAdvType()}' в GRuT")
            }
        }

        val clientId = checkAndCreateGrutClient(campaignId)
        createGrutAssets(prodTCampaign, targetTCampaign, clientId)

        if (targetTCampaign != null) {
            updateGrutCampaign(prodTCampaign, clientId)
        } else {
            createGrutCampaign(prodTCampaign, clientId)
        }

        var resultText = "Кампания $campaignId была успешно скопирована"
        if (param.queueTaskToUpdateAdsJob) {
            uacBannerService.updateAdsDeferred(ClientId.fromLong(clientId), param.operator.uid, campaignId.toString())
            resultText += " и поставлена в очередь на перегенерацию объявлений"
        }

        return InternalToolResult(resultText)
    }

    /**
     * Проверяет есть ли клиент на DT/TS в груте и если нет - создает
     */
    private fun checkAndCreateGrutClient(campaignId: Long): Long {
        val clientIdByCampaignId = shardHelper.getClientIdsByCampaignIds(setOf(campaignId))
        if (clientIdByCampaignId.isNullOrEmpty()) {
            throw InternalToolValidationException("Нет кампании в mysql базе на DT/TS. " +
                "Перед копированием заявки пожалуйста воспользуйтесь телепортом mysql кампании")
        }
        val clientId = clientIdByCampaignId.values.first()

        val targetTClient = grutApiService.clientGrutDao.getClient(clientId)
        if (targetTClient == null) {
            val targetDirectClient = clientService.getClient(ClientId.fromLong(clientId))
                ?: throw InternalToolValidationException("В mysql нет клиента $clientId на DT/TS")
            createGrutClient(targetDirectClient)
        }
        return clientId
    }

    /**
     * Удаляет текущие ассеты на кампании в DT/TS (если есть) и копирует новые из прода
     */
    private fun createGrutAssets(
        prodTCampaign: Schema.TCampaign,
        targetTCampaign: Schema.TCampaign?,
        clientId: Long,
    ) {
        val prodAssetIds = prodTCampaign.spec.briefAssetLinks.linksList
            .map { it.assetId }
        val prodAssets = prodGrutApiService.assetGrutApi.getAssets(prodAssetIds)

        val targetAssetIds = targetTCampaign?.spec?.briefAssetLinks?.linksList
            ?.map { it.assetId }
        if (!targetAssetIds.isNullOrEmpty()) {
            grutTransactionProvider.runRetryable(3) {
                grutApiService.assetGrutApi.deleteObjects(targetAssetIds)
            }
        }

        val assetsToCreate = prodAssets
            .map { toAssetGrut(it, clientId) }
        grutApiService.assetGrutApi.createObjects(assetsToCreate)
    }

    private fun updateGrutCampaign(
        tCampaign: Schema.TCampaign,
        clientId: Long,
    ) {
        grutApiService.briefGrutApi.updateBriefFull(Schema.TCampaign.newBuilder()
            .apply {
                meta = Schema.TCampaignMeta.newBuilder()
                    .setId(tCampaign.meta.id)
                    .setClientId(clientId)
                    .build()
                spec = tCampaign.spec
            }.build())
    }

    private fun createGrutCampaign(
        tCampaign: Schema.TCampaign,
        targetClientId: Long,
    ) {
        grutApiService.briefGrutApi.createBrief(Schema.TCampaign.newBuilder()
            .apply {
                meta = Schema.TCampaignMeta.newBuilder()
                    .apply {
                        id = tCampaign.meta.id
                        campaignType = tCampaign.meta.campaignType.toAdvType()!!.toEAdCampaignType()
                        clientId = targetClientId
                    }.build()
                spec = tCampaign.spec
            }.build()
        )
    }

    private fun createGrutClient(client: Client) {
        grutApiService.clientGrutDao.createOrUpdateClient(
            ClientGrutModel(client = Client()
                .withId(client.id)
                .withChiefUid(client.chiefUid)
                .withName(client.name)
                .withCreateDate(client.createDate),
                ndsHistory = listOf()
            )
        )
    }
}

