package ru.yandex.direct.internaltools.tools.uac

import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbCampaignRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbDirectCampaignRepository
import ru.yandex.direct.core.entity.uac.service.UacBannerService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.internaltools.core.BaseInternalTool
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup
import ru.yandex.direct.internaltools.core.annotations.tool.Action
import ru.yandex.direct.internaltools.core.annotations.tool.Category
import ru.yandex.direct.internaltools.core.annotations.tool.Tool
import ru.yandex.direct.internaltools.core.container.InternalToolResult
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole
import ru.yandex.direct.internaltools.core.enums.InternalToolAction
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory
import ru.yandex.direct.internaltools.core.enums.InternalToolType
import ru.yandex.direct.internaltools.tools.uac.model.UpdateAdsParam
import ru.yandex.direct.internaltools.utils.ToolParameterUtils
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.constraint.CommonConstraints.notNull
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

private const val DESCRIPTION =
    """ По заданному списку cid кампаний поставить в очередь задание на обновление объявлений
"""
const val CAMPAIGNID_NOT_EXISTS = "Кампании с ID %s не являются UAC кампаниями"
const val UPDATE_CAMPAIGNID_ADS_DEFFERED = "Кампании с ID %s поставлены в очередь на обновление объявлений"

@Tool(
    name = "Поставить в очередь задание на обновление объявлений",
    label = "update_ads_deferred_tool",
    description = DESCRIPTION,
    consumes = UpdateAdsParam::class,
    type = InternalToolType.WRITER
)
@Action(InternalToolAction.ADD)
@Category(InternalToolCategory.UAC)
@AccessGroup(InternalToolAccessRole.SUPER)
class UpdateAdsDeferredTool(
    private val shardHelper: ShardHelper,
    private val uacBannerService: UacBannerService,
    private val uacYdbDirectCampaignRepository: UacYdbDirectCampaignRepository,
    private val uacYdbCampaignRepository: UacYdbCampaignRepository,
) : BaseInternalTool<UpdateAdsParam> {

    override fun validate(updateAdsParam: UpdateAdsParam): ValidationResult<UpdateAdsParam, Defect<Any>> {
        val vb: ItemValidationBuilder<UpdateAdsParam, Defect<Any>> = ItemValidationBuilder.of(updateAdsParam)
        vb.item(updateAdsParam.campaignIds, "Список ID кампаний")
            .check(ToolParameterUtils.isStringWithValidLongIds(1))
            .check(notNull())
        return vb.result
    }

    override fun process(updateAdsParam: UpdateAdsParam): InternalToolResult {
        val foundCampaignIds = mutableSetOf<Long>()
        val notFoundCampaignIds = mutableSetOf<Long>()
        val campaignIds = ToolParameterUtils.getLongIdsFromString(updateAdsParam.campaignIds)
        val campaignIdsYDB = uacYdbDirectCampaignRepository.getCampaignIdsByDirectCampaignIds(campaignIds)

        shardHelper.getClientIdsByCampaignIds(campaignIds)
            .forEach { (campaignId, clientId) ->
                if (campaignIdsYDB.containsKey(campaignId)) {
                    foundCampaignIds.add(campaignId)
                    val uacCampaignId = campaignIdsYDB[campaignId]!!
                    uacYdbCampaignRepository.updateBriefSynced(uacCampaignId, false)
                    uacBannerService.updateAdsDeferred(ClientId.fromLong(clientId), updateAdsParam.operator.uid, uacCampaignId)
                } else {
                    notFoundCampaignIds.add(campaignId)
                }
            }

        var message = ""
        if (notFoundCampaignIds.size > 0) {
            message += String.format(CAMPAIGNID_NOT_EXISTS, notFoundCampaignIds.joinToString(",")) + "\n"
        }
        message += String.format(UPDATE_CAMPAIGNID_ADS_DEFFERED, foundCampaignIds.joinToString(","))

        return InternalToolResult(message)
    }
}
