package ru.yandex.direct.internaltools.tools.userblock;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.userblock.model.UserBlockParams;
import ru.yandex.direct.internaltools.tools.userblock.model.UserResult;
import ru.yandex.direct.internaltools.tools.userblock.service.UserBlockToolService;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.defect.CollectionDefects.maxCollectionSize;
import static ru.yandex.direct.validation.defect.CollectionDefects.notEmptyCollection;

@Tool(
        name = "Массовая блокировка пользователей",
        label = "user_block",
        description = "Выполняет массовую блокировку пользователей",
        consumes = UserBlockParams.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.MODERATE)
@AccessGroup({InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPER, InternalToolAccessRole.PLACER})
@ParametersAreNonnullByDefault
public class UserBlockTool extends MassInternalTool<UserBlockParams, UserResult> {
    private static final int USERS_BLOCK_LIMIT = 100;

    private static final Logger logger = LoggerFactory.getLogger(UserBlockTool.class);

    private final UserService userService;

    private final UserBlockToolService userBlockService;

    @Autowired
    public UserBlockTool(UserService userService,
                         UserBlockToolService userBlockService) {
        this.userService = userService;
        this.userBlockService = userBlockService;
    }

    /**
     * В валидации проверяем только корректность ввода
     * Проверки по существу будут выполняться уже при попытке применить
     */
    @Override
    public ValidationResult<UserBlockParams, Defect> validate(UserBlockParams params) {
        Set<String> logins = userBlockService.extractLogins(params.getLogins());
        ItemValidationBuilder<UserBlockParams, Defect> vr = ItemValidationBuilder.of(params);
        vr.item(params.getLogins(), "logins")
                .check(fromPredicate(v -> !logins.isEmpty(), notEmptyCollection()))
                // не позволяем за раз заблокировать много логинов
                .check(fromPredicate(v -> logins.size() <= USERS_BLOCK_LIMIT, maxCollectionSize(USERS_BLOCK_LIMIT)));
        vr.item(params.getBlockReasonType(), "blockReasonType")
                .check(notNull());
        vr.item(params.getBlockComment(), "blockComment")
                .check(maxStringLength(65_535));
        return vr.getResult();
    }

    @Override
    protected List<UserResult> getMassData(UserBlockParams params) {
        User operator = params.getOperator();

        Set<String> inputLogins = userBlockService.extractLogins(params.getLogins());
        checkState(!inputLogins.isEmpty());

        // Нам нужно заблокировать не только указанных пользователей, но также и всех пользователей с такими же clientId
        List<User> inputUsers = userService.massGetUserByLogin(inputLogins);
        Map<ClientId, User> inputUsersByClientId = StreamEx.of(inputUsers).toMap(User::getClientId, u -> u);
        Set<ClientId> clientIds = StreamEx.of(inputUsers).map(User::getClientId).toSet();
        Map<ClientId, List<Long>> clientsUids = userService.massGetUidsByClientIds(clientIds);
        Set<Long> allUidsToBlock = EntryStream.of(clientsUids).values().flatMap(Collection::stream).collect(toSet());
        List<User> users = userService.massGetUser(allUidsToBlock);
        Map<String, User> usersByLogin = StreamEx.of(users).toMap(User::getLogin, identity());

        Map<Long, List<Long>> campaignIdsByClientId = userService.getCampaignIdsForStopping(clientIds);

        // Общий набор результатов для всех логинов, участвующих в операции
        Map<String, UserResult> resultsMap = userBlockService.initResults(inputLogins, usersByLogin);
        userBlockService.addLinkedUsersComments(inputLogins, inputUsersByClientId, usersByLogin, resultsMap);

        // Выполняем проверки, если находим ошибки, возвращаем результат, не применяя изменений в базу
        if (userBlockService.hasNotFoundUsers(inputLogins, usersByLogin, resultsMap)) {
            return userBlockService.finalizeResultsWithErrors(allUidsToBlock, resultsMap);
        }
        if (params.isNeedTierCheck()) {
            if (userBlockService.hasImportantClients(clientIds, usersByLogin, resultsMap)) {
                return userBlockService.finalizeResultsWithErrors(allUidsToBlock, resultsMap);
            }
        }
        if (params.isNeedCampaignsCountCheck()) {
            if (userBlockService.hasClientsWithManyActiveCampaigns(usersByLogin, campaignIdsByClientId, resultsMap)) {
                return userBlockService.finalizeResultsWithErrors(allUidsToBlock, resultsMap);
            }
        }

        // Если все проверки пройдены, зовём метод, который будет выполнять собственно блокировку пользователей
        List<User> usersToBlock = StreamEx.of(users).filter(u -> !u.getStatusBlocked()).toList();
        if (usersToBlock.isEmpty()) {
            logger.info("No users to block");
            return userBlockService.finalizeResultsSuccessfully(allUidsToBlock, resultsMap);
        }
        var vr = userService.blockUsers(operator, usersToBlock, campaignIdsByClientId,
                params.getBlockReasonType(), params.getBlockComment(), params.isNeedCampaignsCountCheck());

        // По идее, после проверок выше операция не должна возвращать ошибки валидации, но если вдруг такое случится
        // (например, если там добавят какие-то дополнительные проверки), то мы вытащим дефекты как toString
        if (vr.hasAnyErrors()) {
            userBlockService.processValidationResultErrors(vr, usersByLogin, resultsMap);
            return userBlockService.finalizeResultsWithErrors(allUidsToBlock, resultsMap);
        }

        return userBlockService.finalizeResultsSuccessfully(allUidsToBlock, resultsMap);
    }
}
