package ru.yandex.direct.internaltools.tools.userblock;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.implementations.MassInternalTool;
import ru.yandex.direct.internaltools.tools.userblock.model.UserResult;
import ru.yandex.direct.internaltools.tools.userblock.model.UserUnblockParams;
import ru.yandex.direct.internaltools.tools.userblock.service.UserBlockToolService;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.defect.CollectionDefects.maxCollectionSize;
import static ru.yandex.direct.validation.defect.CollectionDefects.notEmptyCollection;

@Tool(
        name = "Массовая разблокировка пользователей",
        label = "user_unblock",
        description = "Выполняет массовую разблокировку пользователей",
        consumes = UserUnblockParams.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@Category(InternalToolCategory.MODERATE)
@AccessGroup({InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPER, InternalToolAccessRole.PLACER})
@ParametersAreNonnullByDefault
public class UserUnblockTool extends MassInternalTool<UserUnblockParams, UserResult> {
    private static final Logger logger = LoggerFactory.getLogger(UserUnblockTool.class);

    private static final int USERS_UNBLOCK_LIMIT = 100;

    private final UserService userService;
    private final UserBlockToolService userBlockService;

    @Autowired
    public UserUnblockTool(UserService userService,
                           UserBlockToolService userBlockService) {
        this.userService = userService;
        this.userBlockService = userBlockService;
    }

    @Override
    public ValidationResult<UserUnblockParams, Defect> validate(UserUnblockParams params) {
        Set<String> logins = userBlockService.extractLogins(params.getLogins());
        ItemValidationBuilder<UserUnblockParams, Defect> vr = ItemValidationBuilder.of(params);
        vr.item(params.getLogins(), "logins")
                .check(fromPredicate(v -> !logins.isEmpty(), notEmptyCollection()))
                // не позволяем за раз разблокировать много логинов
                .check(fromPredicate(v -> logins.size() <= USERS_UNBLOCK_LIMIT,
                        maxCollectionSize(USERS_UNBLOCK_LIMIT)));
        return vr.getResult();
    }

    @Override
    protected List<UserResult> getMassData(UserUnblockParams params) {
        Set<String> inputLogins = userBlockService.extractLogins(params.getLogins());
        checkState(!inputLogins.isEmpty());

        // Нам нужно разблокировать не только указанных пользователей, но также и всех пользователей с такими же
        // clientId
        List<User> inputUsers = userService.massGetUserByLogin(inputLogins);
        Map<ClientId, User> inputUsersByClientId = StreamEx.of(inputUsers).toMap(User::getClientId, u -> u);
        Set<ClientId> clientIds = StreamEx.of(inputUsers).map(User::getClientId).toSet();
        Map<ClientId, List<Long>> clientsUids = userService.massGetUidsByClientIds(clientIds);
        Set<Long> allUidsToUnblock = EntryStream.of(clientsUids).values().flatMap(Collection::stream).collect(toSet());
        List<User> users = userService.massGetUser(allUidsToUnblock);
        Map<String, User> usersByLogin = StreamEx.of(users).toMap(User::getLogin, identity());

        // Общий набор результатов для всех логинов, участвующих в операции
        Map<String, UserResult> resultsMap = userBlockService.initResults(inputLogins, usersByLogin);
        userBlockService.addLinkedUsersComments(inputLogins, inputUsersByClientId, usersByLogin, resultsMap);

        // Выполняем проверки, если находим ошибки, возвращаем результат, не применяя изменений в базу
        if (userBlockService.hasNotFoundUsers(inputLogins, usersByLogin, resultsMap)) {
            return userBlockService.finalizeResultsWithErrors(allUidsToUnblock, resultsMap);
        }

        // Пробуем разблокировать пользователей
        List<User> usersToUnblock = StreamEx.of(users).filter(User::getStatusBlocked).toList();
        if (usersToUnblock.isEmpty()) {
            logger.info("No users to unblock");
            return userBlockService.finalizeResultsSuccessfully(allUidsToUnblock, resultsMap);
        }
        var vr = userService.unblockUsers(users);

        // Найденные дефекты выводим как toString
        if (vr.hasAnyErrors()) {
            userBlockService.processValidationResultErrors(vr, usersByLogin, resultsMap);
            return userBlockService.finalizeResultsWithErrors(allUidsToUnblock, resultsMap);
        }

        return userBlockService.finalizeResultsSuccessfully(allUidsToUnblock, resultsMap);
    }
}
