package ru.yandex.direct.internaltools.tools.userblock.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.entity.user.service.validation.BlockUserValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.internaltools.tools.userblock.model.UserResult;
import ru.yandex.direct.internaltools.tools.userblock.model.UserStatus;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.user.service.validation.BlockUserValidationService.MAX_CAMPAIGNS_COUNT_FOR_STOPPING;
import static ru.yandex.direct.internaltools.tools.userblock.model.UserStatus.BLOCKED;
import static ru.yandex.direct.internaltools.tools.userblock.model.UserStatus.NOT_BLOCKED;
import static ru.yandex.direct.internaltools.tools.userblock.model.UserStatus.NOT_FOUND;

@Service
public class UserBlockToolService {
    private static final Logger logger = LoggerFactory.getLogger(UserBlockToolService.class);

    private final UserService userService;
    private final BlockUserValidationService blockUserValidationService;


    @Autowired
    public UserBlockToolService(UserService userService,
                                BlockUserValidationService blockUserValidationService) {
        this.userService = userService;
        this.blockUserValidationService = blockUserValidationService;
    }

    public Set<String> extractLogins(String loginsText) {
        // Логины можно указывать в столбик, а можно через запятую
        return StreamEx.of(loginsText.split("[,\\n]"))
                .map(String::trim)
                .filter(s -> !s.isEmpty())
                .toSet();
    }

    public Map<String, UserResult> initResults(Set<String> inputLogins, Map<String, User> usersByLogin) {
        Map<String, UserResult> resultsMap = StreamEx.of(inputLogins)
                .append(usersByLogin.keySet())
                .distinct()
                .toMap(identity(), UserResult::new);

        // Устанавливаем statusBefore
        for (String login : resultsMap.keySet()) {
            if (!usersByLogin.containsKey(login)) {
                resultsMap.get(login).setStatusBefore(NOT_FOUND);
            } else {
                User user = usersByLogin.get(login);
                UserStatus status = Boolean.TRUE.equals(user.getStatusBlocked()) ? BLOCKED : NOT_BLOCKED;
                resultsMap.get(login).setStatusBefore(status);
            }
        }

        return resultsMap;
    }

    public void addLinkedUsersComments(Set<String> inputLogins,
                                       Map<ClientId, User> inputUsersByClientId,
                                       Map<String, User> usersByLogin,
                                       Map<String, UserResult> resultsMap) {
        // Устанавливаем комментарии о том, какой пользователь с кем связан (по clientId)
        for (String login : resultsMap.keySet()) {
            if (usersByLogin.containsKey(login) && !inputLogins.contains(login)) {
                User user = usersByLogin.get(login);
                String linkedLogin = inputUsersByClientId.get(user.getClientId()).getLogin();
                String comment = String.format(
                        "Пользователь был добавлен вместе с %s, так как они относятся к одному ClientId", linkedLogin);
                resultsMap.get(login).appendComment(comment);
            }
        }
    }

    public boolean hasNotFoundUsers(Set<String> inputLogins,
                                    Map<String, User> usersByLogin,
                                    Map<String, UserResult> resultsMap) {
        List<String> notFoundLogins = StreamEx.of(inputLogins)
                .filter(login -> !usersByLogin.containsKey(login))
                .toList();
        if (notFoundLogins.isEmpty()) {
            return false;
        }
        for (String login : notFoundLogins) {
            resultsMap.get(login).appendError("Пользователь не найден");
        }
        return true;
    }

    public boolean hasImportantClients(Set<ClientId> clientIds, Map<String, User> usersByLogin,
                                       Map<String, UserResult> resultsMap) {
        Set<ClientId> importantClients = blockUserValidationService.getImportantClients(clientIds);
        if (importantClients.isEmpty()) {
            return false;
        }
        for (String login : resultsMap.keySet()) {
            if (usersByLogin.containsKey(login)) {
                User user = usersByLogin.get(login);
                if (importantClients.contains(user.getClientId())) {
                    resultsMap.get(login).appendError(
                            String.format("Клиент %s определён как important", user.getClientId()));
                }
            }
        }
        return true;
    }

    public boolean hasClientsWithManyActiveCampaigns(Map<String, User> usersByLogin,
                                                     Map<Long, List<Long>> campaignIdsByClientId,
                                                     Map<String, UserResult> resultsMap) {
        Map<Long, Integer> clientsAboveCampCountLimit = EntryStream.of(campaignIdsByClientId)
                .mapValues(List::size)
                .filterValues(campaignsCnt -> campaignsCnt > MAX_CAMPAIGNS_COUNT_FOR_STOPPING)
                .toMap();
        if (clientsAboveCampCountLimit.isEmpty()) {
            return false;
        }
        for (String login : resultsMap.keySet()) {
            User user = usersByLogin.get(login);
            if (user != null && clientsAboveCampCountLimit.containsKey(user.getClientId().asLong())) {
                String error = String.format(
                        "Превышен лимит активных кампаний у клиента: %d (разрешено максимум %d)",
                        clientsAboveCampCountLimit.get(user.getClientId().asLong()),
                        MAX_CAMPAIGNS_COUNT_FOR_STOPPING);
                resultsMap.get(login).appendError(error);
            }
        }
        return true;
    }

    public void processValidationResultErrors(ValidationResult<List<User>, Defect> vr,
                                              Map<String, User> usersByLogin,
                                              Map<String, UserResult> resultsMap) {
        Map<ClientId, List<String>> clientsErrors = extractValidationResultErrors(vr);
        for (String login : resultsMap.keySet()) {
            User user = usersByLogin.get(login);
            if (user != null && clientsErrors.containsKey(user.getClientId())) {
                List<String> errors = clientsErrors.get(user.getClientId());
                UserResult result = resultsMap.get(login);
                for (String error : errors) {
                    result.appendError(error);
                }
            }
        }
    }

    private Map<ClientId, List<String>> extractValidationResultErrors(ValidationResult<List<User>, Defect> vr) {
        Map<ClientId, List<String>> clientIdErrors = new HashMap<>();
        var subResults = vr.getSubResults();
        for (PathNode pathNode : subResults.keySet()) {
            var validationResult = subResults.get(pathNode);
            if (validationResult.hasAnyErrors()) {
                List<Defect> defects = validationResult.getErrors();
                User user = (User) validationResult.getValue();
                ClientId clientId = user.getClientId();
                clientIdErrors.putIfAbsent(clientId, new ArrayList<>());
                clientIdErrors.get(clientId).add(String.format("%s (%s)", user.getLogin(), defects));
            }
        }
        return clientIdErrors;
    }

    public List<UserResult> finalizeResultsWithErrors(Set<Long> allUids,
                                                      Map<String, UserResult> resultsMap) {
        return finalizeResults(allUids, resultsMap, true);
    }

    public List<UserResult> finalizeResultsSuccessfully(Set<Long> allUids,
                                                        Map<String, UserResult> resultsMap) {
        return finalizeResults(allUids, resultsMap, false);
    }

    /**
     * Завершает заполнение результатов перед отдачей пользователю
     */
    private List<UserResult> finalizeResults(Set<Long> allUids,
                                             Map<String, UserResult> resultsMap,
                                             boolean hasErrors) {
        List<User> usersAfter = userService.massGetUser(allUids);
        Map<String, User> usersAfterByLogin = StreamEx.of(usersAfter).toMap(User::getLogin, identity());

        // Устанавливаем statusAfter
        for (String login : resultsMap.keySet()) {
            if (!usersAfterByLogin.containsKey(login)) {
                resultsMap.get(login).setStatusAfter(NOT_FOUND);
            } else {
                User user = usersAfterByLogin.get(login);
                UserStatus status = Boolean.TRUE.equals(user.getStatusBlocked()) ? BLOCKED : NOT_BLOCKED;
                resultsMap.get(login).setStatusAfter(status);
            }
        }
        // Если были найдены ошибки, добавляем комментарий о том, что изменения не применялись
        if (hasErrors) {
            for (String login : resultsMap.keySet()) {
                resultsMap.get(login).appendComment("Изменения не применялись");
            }
        }
        List<UserResult> results = StreamEx.of(resultsMap.values()).toList();
        logger.info("Finalizing results: {}", JsonUtils.toJson(results));
        return results;
    }
}
