package ru.yandex.direct.internaltools.tools.users;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.user.model.UsersOptionsOptsValues;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.tools.users.model.DoNotSendSmsAction;
import ru.yandex.direct.internaltools.tools.users.model.DoNotSendSmsInput;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

@Tool(
        consumes = DoNotSendSmsInput.class,
        label = "do_not_send_sms",
        name = "Запрет отправки СМС",
        description = "Позволяет поставить или снять запрет на отправку СМС конкретному пользователю либо по UID," +
                " либо по логину. Отключается только сама отправка, " +
                "очередь СМС во всём остальном будет обработана.",
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.UPDATE)
@AccessGroup({InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER})
@Category(InternalToolCategory.OTHER)
@ParametersAreNonnullByDefault
public class DoNotSendSmsTool implements BaseInternalTool<DoNotSendSmsInput> {
    private static final Logger LOGGER = LoggerFactory.getLogger(DoNotSendSmsTool.class);

    private final UserRepository repository;
    private final ShardHelper shardHelper;

    @Autowired
    public DoNotSendSmsTool(UserRepository repository, ShardHelper shardHelper) {
        this.repository = repository;
        this.shardHelper = shardHelper;
    }

    @Override
    @SuppressWarnings("rawtypes")
    public ValidationResult<DoNotSendSmsInput, Defect> validate(DoNotSendSmsInput input) {
        ItemValidationBuilder<DoNotSendSmsInput, Defect> vb = ItemValidationBuilder.of(input);
        vb.item(input.getAction(), DoNotSendSmsInput.ACTION_LABEL)
                .check(CommonConstraints.notNull());
        vb.item(input.getLogin(), DoNotSendSmsInput.LOGIN_LABEL)
                .check(CommonConstraints.notNull(), When.isTrue(input.getUid() == null))
                .check(CommonConstraints.isNull(), When.isTrue(input.getUid() != null))
                .check(Constraint.fromPredicate(login ->
                                repository.usersExistByLogins(List.of(login)).contains(login),
                                    CommonDefects.objectNotFound()),
                        When.notNull());
        vb.item(input.getUid(), DoNotSendSmsInput.UID_LABEL) // сочетание с логином проверили выше
                .check(Constraint.fromPredicate(shardHelper::isExistentUid, CommonDefects.objectNotFound()),
                        When.notNull());
        return vb.getResult();
    }

    @Override
    public InternalToolResult process(DoNotSendSmsInput input) {
        var uid = input.getUid();
        var login = input.getLogin();
        if (login == null) {
            login = shardHelper.getLoginByUid(uid);
        } else {
            uid = shardHelper.getUidByLogin(login);
        }

        var shard = shardHelper.getShardByClientUidStrictly(uid);
        var flags = new HashSet<>(repository.getOpts(shard, uid));

        switch (input.getAction()) {
            case FORBID:
                flags.add(UsersOptionsOptsValues.DO_NOT_SEND_SMS);
                break;
            case ALLOW:
                flags.remove(UsersOptionsOptsValues.DO_NOT_SEND_SMS);
                break;
            case READ:
                break;
            default:
                return new InternalToolResult("Action " + input.getAction() + " is not supported");
        }
        var result = String.format("UID: %d, логин: %s, %s", uid, login, flagToText(flags));
        if (!input.getAction().equals(DoNotSendSmsAction.READ)) {
            LOGGER.info(result);
            repository.setOpts(shard, uid, flags);
        }
        return new InternalToolResult(result);
    }

    private String flagToText(Set<UsersOptionsOptsValues> values) {
        return values.contains(UsersOptionsOptsValues.DO_NOT_SEND_SMS) ? "отправка запрещена" : "отправка разрешена";
    }
}
