package ru.yandex.direct.internaltools.tools.userssegments;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiConsumer;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.adgroup.model.InternalStatus;
import ru.yandex.direct.core.entity.adgroup.model.UsersSegment;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.userssegments.repository.UsersSegmentRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.internaltools.core.BaseInternalTool;
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup;
import ru.yandex.direct.internaltools.core.annotations.tool.Action;
import ru.yandex.direct.internaltools.core.annotations.tool.Category;
import ru.yandex.direct.internaltools.core.annotations.tool.Tool;
import ru.yandex.direct.internaltools.core.container.InternalToolResult;
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole;
import ru.yandex.direct.internaltools.core.enums.InternalToolAction;
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory;
import ru.yandex.direct.internaltools.core.enums.InternalToolType;
import ru.yandex.direct.internaltools.core.exception.InternalToolValidationException;
import ru.yandex.direct.internaltools.tools.userssegments.model.AdShowType;
import ru.yandex.direct.internaltools.tools.userssegments.model.UsersSegmentsUpdateTimeResetParameters;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.internaltools.tools.userssegments.model.UsersSegmentsUpdateTimeResetParameters.ADGROUPS_LABEL;
import static ru.yandex.direct.utils.StringUtils.joinLongsToString;
import static ru.yandex.direct.utils.StringUtils.parseLongsFromString;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;

@Tool(
        name = "Сегменты пользователей на основе видео/аудио/ТГО групп объявлений. Сброс update time.",
        label = "video_segment_goals_last_successful_update_time_reset",
        description = "Позволяет легко для сегментов из video_segment_goals менять дату, за которую были прочитаны " +
                "логи. Это поволяет пересобирать заново отдельные сегменты, если логи БК за прошедший период " +
                "оказались неполными или собирать сегмент за прошедший период, когда клиент забыл или случайно снял " +
                "галочку. Если хотя бы одна из найденных строк не может быть обновленна - вся операция не выполняется.",
        consumes = UsersSegmentsUpdateTimeResetParameters.class,
        type = InternalToolType.WRITER
)
@Action(InternalToolAction.EXECUTE)
@Category(InternalToolCategory.OTHER)
@AccessGroup({InternalToolAccessRole.DEVELOPER})
public class UsersSegmentsUpdateTimeResetTool implements
        BaseInternalTool<UsersSegmentsUpdateTimeResetParameters> {

    // ограничим максимальную длинну, чтобы случайно очень сильно не грузануть базу
    // наверное классно было бы перенести это в аннотацию
    // ru.yandex.direct.internaltools.core.annotations.input.Input
    private static final int ADGROUPS_LABEL_MAX_LEGNTH = 1000;

    @Autowired
    ShardHelper shardHelper;

    @Autowired
    UsersSegmentRepository usersSegmentRepository;

    @Autowired
    DslContextProvider dslContextProvider;

    @Autowired
    AdGroupRepository adGroupRepository;

    @Override
    public InternalToolResult process(UsersSegmentsUpdateTimeResetParameters parameters) {
        List<Long> adGroups = parseLongsFromString(parameters.getAdGroups());
        List<Integer> shards = shardHelper.getShardsByGroupId(adGroups);

        validateAllAdGroupsExists(shards, adGroups);

        Map<Integer, List<Long>> adGroupIdsByShard = getShardWithAdGroupIdStream(shards, adGroups)
                .grouping();

        processAdGroupIds(adGroupIdsByShard, (ctx, adGroupId) ->
                validateAdGroupUsersSegments(ctx, adGroupId, parameters.getAdShowType()));

        Integer[] totalRows = new Integer[1];
        totalRows[0] = 0;
        processAdGroupIds(adGroupIdsByShard, (ctx, adGroupId) ->
                totalRows[0] += updateAdGroupGoals(ctx, adGroupId, parameters.getAdShowType(),
                        parameters.getNewLastSuccessfulUpdateTime()));

        return new InternalToolResult().withMessage("Обновленно строк: " + totalRows[0]);
    }

    private static EntryStream<Integer, Long> getShardWithAdGroupIdStream(List<Integer> shards, List<Long> adGroups) {
        return StreamEx.of(shards).zipWith(adGroups.stream());
    }

    private void processAdGroupIds(Map<Integer, List<Long>> adGroupIdsByShard,
                                   BiConsumer<DSLContext, Long> adGroupIdProcessor) {
        adGroupIdsByShard.forEach((shard, adGroupIds) ->
                dslContextProvider.ppc(shard).transaction(conf ->
                        adGroupIds.forEach(adGroupId -> adGroupIdProcessor.accept(conf.dsl(), adGroupId))));
    }

    private void validateAllAdGroupsExists(List<Integer> shards, List<Long> adGroups) {
        List<Long> notFoundAdGroupIds = getShardWithAdGroupIdStream(shards, adGroups)
                .filterKeys(Objects::isNull)
                .values()
                .toList();
        if (notFoundAdGroupIds.size() > 0) {
            throw new InternalToolValidationException("Группы не найдены: " + joinLongsToString(notFoundAdGroupIds));
        }
    }

    private void validateAdGroupUsersSegments(DSLContext ctx, Long adGroupId, AdShowType adShowType) {
        for (var goalType : adShowType.getCoreTypes()) {
            UsersSegment segment = usersSegmentRepository.getSegmentByPrimaryKey(ctx, adGroupId, goalType);
            if (segment == null) {
                if (adShowType == AdShowType.ALL) {
                    // когда просят обновить для всех типов это ок, что некоторых может не быть
                    continue;
                } else {
                    throw new InternalToolValidationException(String.format(
                            "Не найден сегмент пользователей с ключом adgroupId %s type %s", adGroupId, goalType));
                }
            }
            if (segment.getInternalStatus() != InternalStatus.COMPLETE) {
                throw new InternalToolValidationException(String.format("Невозможно обновить сегмент пользователей " +
                                "с ключом adgroupId %s type %s, потому что internalStatus этого сегмента %s, " +
                                "но разрешенно только COMPLETE",
                        adGroupId, goalType, segment.getInternalStatus()));
            }
            if (segment.getExternalAudienceId() == 0) {
                throw new InternalToolValidationException(String.format("Невозможно обновить сегмент пользователей " +
                                "с ключом adgroupId %s type %s, потому что его externalAudienceId == 0.",
                        adGroupId, goalType));
            }
        }
    }

    private int updateAdGroupGoals(DSLContext ctx, Long adGroupId, AdShowType adShowType,
                                   LocalDateTime newLastSuccessfulUpdateTime) {
        int totalRows = 0;
        for (var goalType : adShowType.getCoreTypes()) {
            totalRows += usersSegmentRepository.updateSegmentLastSuccessfulUpdateTimeAndResetSegmentErrorCount(ctx,
                    adGroupId, goalType, newLastSuccessfulUpdateTime);
        }
        return totalRows;
    }

    @Override
    public ValidationResult<UsersSegmentsUpdateTimeResetParameters, Defect> validate(
            UsersSegmentsUpdateTimeResetParameters parameters) {
        ItemValidationBuilder<UsersSegmentsUpdateTimeResetParameters, Defect>
                validationBuilder = ItemValidationBuilder.of(parameters);
        validationBuilder.item(parameters.getAdGroups(), ADGROUPS_LABEL)
                .check(notNull())
                .check(maxStringLength(ADGROUPS_LABEL_MAX_LEGNTH));
        return validationBuilder.getResult();
    }

}
