package ru.yandex.direct.internaltools.tools.yql

import java.time.LocalDateTime
import java.time.ZoneId
import java.time.format.DateTimeFormatter
import java.time.temporal.ChronoUnit
import ru.yandex.direct.internaltools.core.BaseInternalTool
import ru.yandex.direct.internaltools.core.annotations.tool.AccessGroup
import ru.yandex.direct.internaltools.core.annotations.tool.Action
import ru.yandex.direct.internaltools.core.annotations.tool.Category
import ru.yandex.direct.internaltools.core.annotations.tool.Tool
import ru.yandex.direct.internaltools.core.container.InternalToolMassResult
import ru.yandex.direct.internaltools.core.container.InternalToolResult
import ru.yandex.direct.internaltools.core.enums.InternalToolAccessRole
import ru.yandex.direct.internaltools.core.enums.InternalToolAction
import ru.yandex.direct.internaltools.core.enums.InternalToolCategory
import ru.yandex.direct.internaltools.core.enums.InternalToolType
import ru.yandex.direct.utils.DateTimeUtils
import ru.yandex.direct.yql.client.YqlClient
import ru.yandex.direct.yql.client.model.OperationResult
import ru.yandex.direct.yql.client.model.OperationResultResponse
import ru.yandex.direct.ytwrapper.model.YtCluster

@Tool(
    name = "Статистика YQL запросов",
    label = "yql_resources_stat",
    description = "Показывается потребление вычислительных ресурсов YQL запросов.\n" +
        "Полный список сортируется по времени, а топ-20 сортируется по количеству YT-операций и CPU.\n" +
        "Внимание стоит обратить:\n" +
        "  - число операций -- больше 50 операций уже не типично;\n" +
        "  - cpu -- гарантировано около 1500 ядер;\n" +
        "  - количество таких же запросов -- есть ли обоснования для частого запуска или, например, есть ошибка, приводящая к перезапуску/запуск запроса в цикле и тп;\n" +
        "  - длительность запроса -- более 2-3 часов считать подозрительным.",
    consumes = YqlStatParam::class,
    type = InternalToolType.REPORT
)
@Action(InternalToolAction.SHOW)
@Category(InternalToolCategory.OTHER)
@AccessGroup(InternalToolAccessRole.SUPER, InternalToolAccessRole.DEVELOPER, InternalToolAccessRole.SUPERREADER)
class YqlResourcesStatTool(private val yqlClient: YqlClient) : BaseInternalTool<YqlStatParam> {

    companion object {
        private const val TOP_COUNT = 20

        private val UTC_ZONE = ZoneId.of("UTC")
        private val DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss'Z'")
        private const val YQL_LINK_PREFIX = "https://yql.yandex-team.ru/Operations/"
        private const val UNKNOWN_VALUE = "unknown"
        private const val RUNNING_STATUS = "RUNNING"
    }

    override fun process(parameters: YqlStatParam): InternalToolResult {
        val result = execute(parameters.cluster, parameters.from, parameters.to)
        val queriesCount = result.groupingBy { it.title }.eachCount()
        val updatedResult = result.map { it.copy(count = queriesCount[it.title]) }
        return if (parameters.needOnlyTop) {
            val running = updatedResult.groupBy { it.status }[RUNNING_STATUS]
            val sortedResult = updatedResult.sortedWith(
                compareByDescending<YqlStatResult> { it.operationsCount }.thenByDescending { it.cpu }
            )
            InternalToolMassResult(sortedResult.take(TOP_COUNT) + (running ?: emptyList()))
        } else {
            val sortedResult = updatedResult.sortedWith(compareBy { it.createdAt })
            InternalToolMassResult(sortedResult)
        }
    }

    fun execute(ytCluster: YtCluster, from: LocalDateTime, to: LocalDateTime): MutableList<YqlStatResult> {
        val utcFromZoned = from.atZone(DateTimeUtils.MSK).withZoneSameInstant(UTC_ZONE)
        val utcToZoned = to.atZone(DateTimeUtils.MSK).withZoneSameInstant(UTC_ZONE)

        val formattedFrom = DATE_TIME_FORMATTER.format(utcFromZoned)
        val formattedTo = DATE_TIME_FORMATTER.format(utcToZoned)

        val results = mutableListOf<YqlStatResult>()
        var page = 0
        val user = yqlClient.getUser()
        val cluster = ytCluster.getName().lowercase()
        while (true) {
            val filters = "(username=$user),(createdAt>=$formattedFrom),(createdAt<=$formattedTo)"
            val operations = yqlClient.operations(filters, page)
            if (operations.page.count == 0) {
                break
            } else {
                page++
                results.addAll(operations.result.mapNotNull { toResult(cluster, it) })
            }
        }
        return results
    }

    private fun toResult(cluster: String, result: OperationResult): YqlStatResult? {
        val title = result.title ?: UNKNOWN_VALUE
        val status = result.status
        val createdAt = result.createdAt.atZone(UTC_ZONE).withZoneSameInstant(DateTimeUtils.MSK).toLocalDateTime()
        val updatedAt = result.updatedAt.atZone(UTC_ZONE).withZoneSameInstant(DateTimeUtils.MSK).toLocalDateTime()
        val duration = ChronoUnit.MINUTES.between(result.createdAt, result.updatedAt).toInt()
        val link = YQL_LINK_PREFIX + result.id
        if (result.status == RUNNING_STATUS) {
            return YqlStatResult(
                title = title,
                status = status,
                createdAt = createdAt,
                updatedAt = updatedAt,
                duration = duration,
                link = link,
                operationsCount = -1,
                cpu = -1
            )
        }
        val opResult = yqlClient.getOperationResult(result.id)
        val ytCluster = getCluster(opResult)
        if (cluster != ytCluster) {
            return null
        }
        val operationsCount = getOperationsCount(opResult)
        val cpu = getCpu(opResult)
        return YqlStatResult(
            title = title,
            status = status,
            createdAt = createdAt,
            updatedAt = updatedAt,
            duration = duration,
            link = link,
            operationsCount = operationsCount,
            cpu = cpu
        )
    }

    private fun getCluster(opResult: OperationResultResponse): String {
        val providers = opResult.plan?.detailed?.providers
        return try {
            providers?.get(0)?.cluster
        } catch (e: RuntimeException) {
            null
        } ?: UNKNOWN_VALUE
    }

    private fun getOperationsCount(body: OperationResultResponse): Int {
        return body.plan?.basic?.nodes?.filter { n -> "op" == n.type }?.count() ?: 0
    }

    private fun getCpu(body: OperationResultResponse): Int {
        return body.progress?.values?.map { it.total ?: 0 }?.sum() ?: 0
    }
}

